"""
Copyright (c) 2011 Idiap Research Institute, http://www.idiap.ch/
Written by Carl Scheffler <carl.scheffler@gmail.com>

This file is part of FaceColorModel.

FaceColorModel is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License version 3 as
published by the Free Software Foundation.

FaceColorModel is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with FaceColorModel. If not, see <http://www.gnu.org/licenses/>.
"""
from __future__ import division
import scipy
import opencv
from opencv import highgui
from PIL import Image

def rgb2ypbpr(iRgb):
    """
    Convert a color vector or matrix containing color vectors from RGB
    to YPbPr color space. The input should be either a length-3 vector
    or a 3 x N matrix. RGB values should be in the range [0,1]. The
    function output has the same shape as the input with YPbPr values
    in the range [0,1].
    """
    A = scipy.array([[ 0.299,     0.587,     0.114],
                     [-0.168736, -0.331264,  0.5],
                     [ 0.5,      -0.418688, -0.081312]])
    b = scipy.array([0, 0.5, 0.5])
    assert (len(iRgb.shape) in [1,2]) and (iRgb.shape[0] == 3), \
           "Expecting a length-3 vector or a matrix with 3 rows."
    if len(iRgb.shape) == 1:
        return scipy.dot(A, iRgb) + b
    else:
        return scipy.dot(A, iRgb) + b[:,scipy.newaxis]

def ypbpr2rgb(iYpbpr):
    """
    Convert a color vector or matrix containing color vectors from YPbPr
    to RGB color space. The input should be either a length-3 vector
    or a 3 x N matrix. YPbPr values should be in the range [0,1]. The
    function output has the same shape as the input with RGB values
    in the range [0,1].
    """
    from scipy.linalg import inv
    A = inv(scipy.array([[ 0.299,     0.587,     0.114],
                         [-0.168736, -0.331264,  0.5],
                         [ 0.5,      -0.418688, -0.081312]]))
    b = scipy.array([0, 0.5, 0.5])
    assert (len(iYpbpr.shape) in [1,2]) and (iYpbpr.shape[0] == 3), \
           "Expecting a length-3 vector or a matrix with 3 rows."
    if len(iYpbpr.shape) == 1:
        return scipy.dot(A, iYpbpr-b)
    else:
        return scipy.dot(A, iYpbpr-b[:,scipy.newaxis])

def load_image(iFilename, iIsColor=None):
    """
    This simply calls the OpenCv function cvLoadImage, *unless* the
    input image is in GIF format since OpenCv does not read GIF. In
    this case, the Python Image Library is used to read the image and
    the result is converted to an IplImage. See the cvLoadImage
    documentation for a description of the input arguments.
    """
    extension = iFilename[iFilename.rfind('.'):].lower()
    if extension == '.gif':
        with open(iFilename, 'rb') as fp:
            try:
                image = Image.open(fp)
                image.load()
            except IOError:
                return None
        if iIsColor == highgui.CV_LOAD_IMAGE_GRAYSCALE:
            image = image.convert('L')
        else:
            image = image.convert('RGB')
        image = opencv.adaptors.PIL2Ipl(image)
    else:
        if iIsColor is None:
            image = highgui.cvLoadImage(iFilename)
        else:
            image = highgui.cvLoadImage(iFilename, iIsColor)
    return image
