/**
   Class for face color modelling and adaptation. Explicitly models
   and adapts color distributions for skin, hair, clothing and
   background around a face. Color models can be used to segmented a
   target image into face, hair, clothing and background regions.

   @author Carl Scheffler (carl.scheffler@gmail.com)
 */

#ifndef __FACECOLORMODEL_H__
#define __FACECOLORMODEL_H__

#include <cv.h>

namespace FaceColorModel {

  /// Indices for the different color models
  const int FCM_NUM_CHANNELS       = 4;
  const int FCM_CHANNEL_SKIN       = 0;
  const int FCM_CHANNEL_HAIR       = 1;
  const int FCM_CHANNEL_CLOTHES    = 2;
  const int FCM_CHANNEL_BACKGROUND = 3;

  /// The two types of color models: discrete histograms over RGB colors
  /// and normal distributions in YPbPr space.
  const int FCM_DISCRETE   = 0;
  const int FCM_CONTINUOUS = 1;

  /// The number of bins along each dimension of color histograms.
  const int FCM_HISTOGRAM_BINS = 16;
  const int FCM_HISTOGRAM_SIZE =
    FCM_HISTOGRAM_BINS * FCM_HISTOGRAM_BINS * FCM_HISTOGRAM_BINS;

  /// The size of the PIM prior and consequently all scale normalized
  /// images in the model [pixels].
  const int FCM_SCALED_SIZE = 128;

  class DiscreteColorPrecompute {
  public:
    double digammaDistribution [FCM_HISTOGRAM_SIZE];
    double logConstant;
  };

  class ContinuousColorPrecompute {
  public:
    double alphaOver2Beta [3];
    double mu [3];
    double logConstant;
  };

  class FaceColorModel {

  // Attributes
  private:

    /// The type of color model (discrete or continuous) for each
    /// channel.
    int mpModelTypes[FCM_NUM_CHANNELS];

    /// The number of variational updates to perform when adapting color
    /// models.
    int mVariationalIterations;

    /// The strength of the Markov random field between adjacent pixels.
    double mKMrf;

    /// Pointers to the prior and posterior color model for each channel.
    double* mpColorPriors[FCM_NUM_CHANNELS];
    double* mpColorPosteriors[FCM_NUM_CHANNELS];

    /// The Probabilistic Index Map prior and posterior along all
    /// channels. Each will contain
    ///   FCM_NUM_CHANNELS * FCM_SCALED_SIZE * FCM_SCALED_SIZE
    /// entries.
    double* mpChannelPrior;
    double* mpChannelPosterior;

    /// Flag: whether the posterior distributions currently equal the
    /// priors.
    bool mPosteriorsEqualPriors;

    /// Pre-computed support variables used during variational inference
    /// log |A|, where A is defined in bgr2ypbpr() in FaceColorModel.cc.
    double mLogDetTransformYpbpr;
    /// Volume of a histogram bin in RGB space.
    double mLogBinVolume;
    /// Log of the channel prior
    double* mpLogChannelPrior;
    /// Pre-computation for color distributions
    void* mpPrecomputeColorPriors [FCM_NUM_CHANNELS];
    void* mpPrecomputeColorPosteriors [FCM_NUM_CHANNELS];

  // Lifecycle methods
  public:

    /// Constructor.
    /// @param iKMrf The strength of the Markov random field between
    ///   adjacent pixels, used during color model adaptation.
    /// @param iSkinType Whether the skin color model is continuous or
    ///   discrete.
    /// @param iHairType Whether the hair color model is continuous or
    ///   discrete.
    FaceColorModel(double iKMrf = 2, int iSkinType = FCM_CONTINUOUS,
		   int iHairType = FCM_CONTINUOUS);

    /// Destructor.
    ~FaceColorModel();


  // Getters and setters
  public:

    /// Pointer to the posterior (post-adaptation) color model for a
    /// particular channel.
    /// @param iIndex The color channel for which to return the
    ///   posterior.
    const double* get_color_posterior(int iChannelIndex);

    /// Pointer to the posterior (post-adaptation) belief over the
    /// channel of each pixel. The double array is indexed as
    /// [channel][x][y] and each entry is a probability such that the
    /// sum over channels equals 1 for each (x,y).
    const double* get_class_posterior();

    /// Return FCM_DISCRETE if the channel has a discrete color
    /// distribution and FCM_CONTINUOUS otherwise.
    /// @param iIndex The color channel for which to return the
    ///   posterior.
    int get_channel_type(int iChannelIndex);


  // General methods
  public:

    /// Adapt the color models from their priors using an input image.
    /// @param ipImage The image to which to adapt. This should be
    ///   square and of size FCM_SCALED_SIZE * FCM_SCALED_SIZE (otherwise
    ///   it will be resized). The image is assumed to contain a face
    ///   found using the Viola-Jones face detector. The bounding box of
    ///   the image should be twice the size of the V-J bounding box.
    ///   Note that the image should have 3 channels of depth 8U (1 byte)
    ///   and should use the BGR color model.
    void adapt_to(IplImage const* ipImage);

    /// Compute the log likelihood of each pixel according to each
    /// posterior (adapted) color model.
    /// @param ipImage The image on which to compute the likelihood.
    /// @param opLogLikelihood Storage for the likelihoods. This is
    ///   indexed as [channel][x][y]. Note that these are *not*
    ///   probabilities.
    void channel_log_likelihood(IplImage const* ipImage,
				double* opLogLikelihood);

    /// Undo any adaptation, reset posterior models to prior models.
    void reset_to_prior();


  // Support methods
  private:

    /// Compute the log likelihood of each color channel and add it to a
    /// given array.
    /// @param ipInputYpbpr A length rows*columns*3 array containing the
    ///   input pixels in YPbPr color space.
    /// @param ipInputBins A length rows*columns array of the color
    ///   histogram bin index of each pixel.
    /// @param opLikelihood The output array to which log likelihoods
    ///   are *added*.
    /// @param iRows The number of rows in the input image.
    /// @param iColumns The number of columns in the input image.
    void __add_color_log_likelihood(
      double* ipInputYpbpr, int* ipInputBins, double* opLikelihood,
      int iRows, int iColumns);

    /// Pre-computation for using a discrete color model to compute
    /// channel likelihoods.
    /// @param ipDistribution A length FCM_HISTOGRAM_SIZE array
    ///   containing the discrete color model.
    /// @param opPrecompute Storage for the pre-computed values.
    void __precompute_discrete_color(
      double const* ipDistribution,
      DiscreteColorPrecompute* opPrecompute);

    /// Pre-computation for using a continuous color model to compute
    /// channel likelihoods.
    /// @param ipDistribution A length 12 array containing the
    ///   continuous color model.
    /// @param opPrecompute Storage for the pre-computed values.
    void __precompute_continuous_color(
      double const* ipDistribution,
      ContinuousColorPrecompute* opPrecompute);

  }; // class FaceColorModel

} // namespace FaceColorModel

#endif
