"""
Use skin images from the Compaq skin database to construct a Dirichlet
prior over the histogram of RGB colors found in skin pixels.
Requires: Compaq skin database at path compaq_database/
Creates:  storage/color_prior_skin_discrete.data

Copyright (c) 2011 Idiap Research Institute, http://www.idiap.ch/
Written by Carl Scheffler <carl.scheffler@gmail.com>

This file is part of FaceColorModel.

FaceColorModel is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License version 3 as
published by the Free Software Foundation.

FaceColorModel is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with FaceColorModel. If not, see <http://www.gnu.org/licenses/>.
"""
from __future__ import division
from scipy import *
from opencv import highgui
import os, sys
from colorlib import load_image
from viola_jones_opencv import viola_jones_opencv
from polya_distribution import compute_polya_parameter

binCount = 16 # Number of histogram bins along each of the RGB dimensions
              # (for a total of binCount**3 bins)
basePath = 'compaq_database/'

# Read command line arguments
verbose = ("--quiet" not in sys.argv[1:])

# Read mask list
maskPath = basePath + 'masks/'
with open(basePath + 'mask-list', 'rt') as fp:
    maskList = [line + '.pbm' for line in fp.read().strip().split('\n')]
maskList.sort()

# Read skin image list
skinPath = basePath + 'skin-images/'
with open(basePath + 'skin-list-good', 'rt') as fp:
    skinList = fp.read().strip().split('\n')
skinList.sort()

# Sanity check that lists match exactly
assert [filename[:filename.find('.')] for filename in maskList] ==\
       [filename[:filename.find('.')] for filename in skinList]

edges = arange(0, 257, 256//binCount)-0.5 # Histogram bin edges
counts = [] # The histogram counts in each image
for i in range(len(maskList)):
    if verbose and ((i+1)%100 == 0):
        print i+1, '/', len(maskList)

    maskImage = load_image(maskPath + maskList[i],
                           highgui.CV_LOAD_IMAGE_GRAYSCALE)
    if maskImage is None:
        if verbose:
            print "Mask is None:", maskList[i]
        continue
    
    grayImage = load_image(skinPath + skinList[i],
                           highgui.CV_LOAD_IMAGE_GRAYSCALE)
    if grayImage is None:
        if verbose:
            print "Skin is None:", skinList[i]
        continue
    
    face = viola_jones_opencv(grayImage)
    if face is None:
        continue
    
    mask = ravel(array(maskImage)[face[0]:face[2], face[1]:face[3]]) > 0
    skin = array(load_image(skinPath + skinList[i]))[face[0]:face[2], face[1]:face[3], :]
    blue  = compress(mask, ravel(skin[:,:,0]))
    if len(blue) == 0: # No skin pixels in mask
        if verbose:
            print 'No skin pixels in image:', skinList[i]
        continue
    green = compress(mask, ravel(skin[:,:,1]))
    red   = compress(mask, ravel(skin[:,:,2]))
    rgb = vstack((red, green, blue))
    histogram, temp = histogramdd(rgb.transpose(), bins=(edges,)*3, normed=False)
    counts.append(ravel(histogram))

if verbose:
    print "Computing Dirichlet prior from RGB histogram counts..."
alpha = compute_polya_parameter(array(counts), verbose=verbose)

if verbose:
    print "Saving..."
with open("storage/color_prior_skin_discrete.data", "wb") as fp:
    fp.write(alpha.data)
