"""
A script for training all of the color models and the PIM prior used
in face color modelling. Run with the --help command line argument for
usage information.

Copyright (c) 2011 Idiap Research Institute, http://www.idiap.ch/
Written by Carl Scheffler <carl.scheffler@gmail.com>

This file is part of FaceColorModel.

FaceColorModel is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License version 3 as
published by the Free Software Foundation.

FaceColorModel is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with FaceColorModel. If not, see <http://www.gnu.org/licenses/>.
"""
from __future__ import division
from scipy import *
import os, sys
import command_line

commandLineSpec = (
    ("inputListFilename", "--input_list", str, "input_list.txt", """
      A text file listing the relative path to each input image
      in the training set."""),
    ("labelListFilename", "--label_list", str, "label_list.txt", """
      A text file listing the relative path to each label image
      in the training set. Each line in the label list should
      correspond to a line in the input list."""),
    ("forcedRetrain", "--force", """
      Force re-training of skin and background color models from
      Compaq database."""),
    ("quiet", "--quiet", """
      Do not print any output to the console."""),
)

# Calling for help
if "--help" in sys.argv[1:]:
    print command_line.usage(commandLineSpec, sys.argv)
    sys.exit()

# Parse command line arguments
try:
    PARAMS, _ = command_line.parse(commandLineSpec, sys.argv)
except ValueError, msg:
    print msg
    print command_line.usage(commandLineSpec, sys.argv)
    sys.exit()

# For skin (continuous and discrete) color models and background
# discrete color model, check if output file already exists and skip
# training (unless --force'd).
for name, dataFile, scriptFile in [
    ("discrete skin",
     "storage/color_prior_skin_discrete.data",
     "train_skin_discrete.py"),
    ("continuous skin",
     "storage/color_prior_skin_continuous.data",
     "train_skin_continuous.py"),
    ("background",
     "storage/color_prior_background_discrete.data",
     "train_background.py")]:

    if not PARAMS['quiet']: print
    try:
        fp = open(dataFile,"rb")
        fp.close()
        if PARAMS['forcedRetrain']:
            retrain = True
        else:
            if not PARAMS['quiet']:
                print "%s color model already exists and will not be retrained."%name.capitalize()
                print 'Delete "%s" to retrain or use the --force command line argument.'%dataFile
            retrain = False
    except IOError:
        retrain = True

    if retrain:
        if not PARAMS['quiet']:
            print "Re-training %s color model."%name
        if PARAMS['quiet']:
            flags = " --quiet"
        else:
            flags = ""
        os.system("python " + scriptFile + flags)


# For hair (continuous and discrete) color models, always retrain.
for model in ['discrete', 'continuous']:
    if not PARAMS['quiet']:
        print
        print "Re-training %s hair color model."%model
    if PARAMS['quiet']:
        flags = " --quiet"
    else:
        flags = ""
    os.system("python train_hair_%s.py --input_list %s --label_list %s"%
              (model, PARAMS['inputListFilename'], PARAMS['labelListFilename']) + flags)


# Clothing model is equivalent to background model
if not PARAMS['quiet']:
    print
    print "Clothing color model is equivalent to background model."


# PIM
if not PARAMS['quiet']:
    print
    print 'Computing PIM prior'
if PARAMS['quiet']:
    flags = " --quiet"
else:
    flags = ""
os.system("python train_pim.py --input_list %s --label_list %s"%
          (PARAMS['inputListFilename'], PARAMS['labelListFilename']) + flags)
