"""
Copyright (c) 2011 Idiap Research Institute, http://www.idiap.ch/
Written by Carl Scheffler <carl.scheffler@gmail.com>

This file is part of FaceColorModel.

FaceColorModel is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License version 3 as
published by the Free Software Foundation.

FaceColorModel is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with FaceColorModel. If not, see <http://www.gnu.org/licenses/>.
"""
from __future__ import division
import scipy as S
from scipy import optimize
from scipy.special import digamma

def compute_normal_gamma_parameters_from_sufficient_stats(iStats):
    """
    Compute the parameters for a normal-gamma distribution by
    computing the maximum likelihood (mean, precision) samples from
    the distribution.
    
    iStats is 6 x N and each row contains [mean, precision] * 3 for
    Y, Pb and Pr channels.
    """
    result = S.empty(12, dtype=float) # [eta, tau, alpha, beta] * 3
    for i in range(3):
        n = iStats.shape[1]
        sumLambda = S.sum(iStats[1+2*i])
        eta = S.sum(iStats[1+2*i] * iStats[0+2*i]) / sumLambda
        tau = n / S.sum(iStats[1+2*i] * (iStats[0+2*i] - eta)**2)
        rhs = S.log(sumLambda/n) - S.sum(S.log(iStats[1+2*i]))/n # Right hand side of alpha equation
        initial = (3-rhs+S.sqrt((3-rhs)**2+24*rhs))/(12*rhs) # initial estimate
        alpha = optimize.fsolve(lambda x, rhs: S.log(x) - digamma(x) - rhs,
                                initial, args=(rhs,))
        beta = n*alpha/sumLambda
        result[i*4:(i+1)*4] = [eta, tau, alpha, beta]
    return result
