
/*
 *  mtp is the ``Multi Tracked Paths'', an implementation of the
 *  k-shortest paths algorithm for multi-target tracking.
 *
 *  Copyright (c) 2012 Idiap Research Institute, http://www.idiap.ch/
 *  Written by Francois Fleuret <francois.fleuret@idiap.ch>
 *
 *  This file is part of mtp.
 *
 *  mtp is free software: you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License version 3 as
 *  published by the Free Software Foundation.
 *
 *  mtp is distributed in the hope that it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 *  License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with selector.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "mtp_graph.h"

// #include <iostream>
#include <float.h>

using namespace std;

class Edge {
public:
  int id, occupied;
  scalar_t length, positivized_length;
  Vertex *origin_vertex, *terminal_vertex;

  // These are the links in the origin_vertex leaving edge list
  Edge *next_leaving_edge, *pred_leaving_edge;

  inline void invert();
};

class Vertex {
public:
  int id;
  Edge *leaving_edges;
  scalar_t distance_from_source;
  Edge *pred_edge_toward_source;

  int last_change; // Used to mark which edges have already been
                   // processed in some methods

  Vertex();

  inline void add_leaving_edge(Edge *e);
  inline void del_leaving_edge(Edge *e);
};

//////////////////////////////////////////////////////////////////////

void Edge::invert() {
  length = - length;
  positivized_length = - positivized_length;
  origin_vertex->del_leaving_edge(this);
  terminal_vertex->add_leaving_edge(this);
  Vertex *t = terminal_vertex;
  terminal_vertex = origin_vertex;
  origin_vertex = t;
}

//////////////////////////////////////////////////////////////////////

Vertex::Vertex() {
  leaving_edges = 0;
}

void Vertex::add_leaving_edge(Edge *e) {
  e->next_leaving_edge = leaving_edges;
  e->pred_leaving_edge = 0;
  if(leaving_edges) { leaving_edges->pred_leaving_edge = e; }
  leaving_edges = e;
}

void Vertex::del_leaving_edge(Edge *e) {
  if(e == leaving_edges) {
    leaving_edges = e->next_leaving_edge;
  }
  if(e->pred_leaving_edge) {
    e->pred_leaving_edge->next_leaving_edge = e->next_leaving_edge;
  }
  if(e->next_leaving_edge) {
    e->next_leaving_edge->pred_leaving_edge = e->pred_leaving_edge;
  }
}

//////////////////////////////////////////////////////////////////////

MTPGraph::MTPGraph(int nb_vertices, int nb_edges,
                   int *vertex_from, int *vertex_to,
                   int source, int sink) {
  _nb_vertices = nb_vertices;
  _nb_edges = nb_edges;

  _edges = new Edge[_nb_edges];
  _vertices = new Vertex[_nb_vertices];
  _front = new Vertex *[_nb_vertices];
  _new_front = new Vertex *[_nb_vertices];

  _source = &_vertices[source];
  _sink = &_vertices[sink];

  for(int k = 0; k < _nb_vertices; k++) {
    _vertices[k].id = k;
  }

  for(int e = 0; e < nb_edges; e++) {
    _vertices[vertex_from[e]].add_leaving_edge(_edges + e);
    _edges[e].occupied = 0;
    _edges[e].id = e;
    _edges[e].origin_vertex = _vertices + vertex_from[e];
    _edges[e].terminal_vertex = _vertices + vertex_to[e];
  }

  paths = 0;
  nb_paths = 0;
}

MTPGraph::~MTPGraph() {
  delete[] _vertices;
  delete[] _edges;
  delete[] _front;
  delete[] _new_front;
  for(int p = 0; p < nb_paths; p++) delete paths[p];
  delete[] paths;
}

//////////////////////////////////////////////////////////////////////

void MTPGraph::print(ostream *os) {
  for(int k = 0; k < _nb_edges; k++) {
    Edge *e = _edges + k;
    (*os) << e->origin_vertex->id
         << " -> "
         << e->terminal_vertex->id
         << " "
         << e->length;
    if(e->occupied) {
      (*os) << " *";
    }
    (*os) << endl;
  }
}

void MTPGraph::print_dot(ostream *os) {
  (*os) << "digraph {" << endl;
  (*os) << "        rankdir=\"LR\";" << endl;
  (*os) << "        node [shape=circle,width=0.75,fixedsize=true];" << endl;
  (*os) << "        edge [color=gray,arrowhead=open]" << endl;
  (*os) << "        " << _source->id << " [peripheries=2];" << endl;
  (*os) << "        " << _sink->id << " [peripheries=2];" << endl;
  for(int k = 0; k < _nb_edges; k++) {
    Edge *e = _edges + k;
    // (*os) << "  " << e->origin_vertex->id << " -> " << e->terminal_vertex->id
    // << ";"
    // << endl;
    (*os) << "        " << e->origin_vertex->id << " -> " << e->terminal_vertex->id
          << " [";
    if(e->occupied) {
      (*os) << "style=bold,color=black,";
    }
    (*os) << "label=\"" << e->length << "\"];" << endl;
  }
  (*os) << "}" << endl;
}

//////////////////////////////////////////////////////////////////////

void MTPGraph::update_positivized_lengths() {
  for(int k = 0; k < _nb_edges; k++) {
    Edge *e = _edges + k;
    e->positivized_length +=
      e->origin_vertex->distance_from_source - e->terminal_vertex->distance_from_source;
  }
}

void MTPGraph::force_positivized_lengths() {
#ifdef VERBOSE
  scalar_t residual_error = 0.0;
  scalar_t max_error = 0.0;
#endif
  for(int k = 0; k < _nb_edges; k++) {
    Edge *e = _edges + k;

    if(e->positivized_length < 0) {

#ifdef VERBOSE
      if((e->origin_vertex->last_change < 0 && e->terminal_vertex->last_change >= 0) ||
         (e->origin_vertex->last_change >= 0 && e->terminal_vertex->last_change < 0)) {
        cout << "Inconsistent non-connexity (this should never happen)." << endl;
        abort();
      }
      if(e->origin_vertex->last_change >= 0 &&
         e->terminal_vertex->last_change >= 0 &&
         e->positivized_length < 0) {
        residual_error -= e->positivized_length;
        max_error = max(max_error, - e->positivized_length);
      }
#endif
      e->positivized_length = 0.0;
    }
  }
#ifdef VERBOSE
  cerr << "residual_error " << residual_error << " max_error " << residual_error << endl;
#endif
}

int MTPGraph::is_dag() {
  Vertex *v;
  Edge *e;

  // We put everybody in the front
  for(int k = 0; k < _nb_vertices; k++) {
    _vertices[k].last_change = -1;
    _front[k] = &_vertices[k];
  }

  int iteration = 0;
  int front_size = _nb_vertices, pred_front_size;

  do {
    // We set the iteration field of all vertex with incoming edges to
    // the current iteration value
    for(int f = 0; f < front_size; f++) {
      v = _front[f];
      for(e = v->leaving_edges; e; e = e->next_leaving_edge) {
        e->terminal_vertex->last_change = iteration;
      }
    }

    pred_front_size = front_size;
    front_size = 0;

    // We remove all the vertices without incoming edge
    for(int f = 0; f < pred_front_size; f++) {
      v = _front[f];
      if(v->last_change == iteration) {
        _front[front_size++] = v;
      }
    }

    iteration++;
  } while(front_size < pred_front_size);

  return front_size == 0;
}

// This method does not change the edge occupation. It only set
// properly for every vertex the fields distance_from_source and
// pred_edge_toward_source.

void MTPGraph::find_shortest_path() {
  Vertex **tmp_front;
  Vertex *v, *tv;
  Edge *e;
  scalar_t d;

#ifdef DEBUG
  if(is_dag()) {
    cout << "find_shortest_path: DAG -> ok" << endl;
  } else {
    for(int e = 0; e < _nb_edges; e++) {
      if(_edges[e].positivized_length < 0)  abort();
    }
    cout << "find_shortest_path: All positivized_length are positive -> ok" << endl;
  }
#endif

  for(int k = 0; k < _nb_vertices; k++) {
    _vertices[k].distance_from_source = FLT_MAX;
    _vertices[k].pred_edge_toward_source = 0;
    _vertices[k].last_change = -1;
  }

  int iteration = 0;

  int front_size = 0, new_front_size;
  _front[front_size++] = _source;
  _source->distance_from_source = 0;

  do {
    new_front_size = 0;

    for(int f = 0; f < front_size; f++) {
      v = _front[f];
      for(e = v->leaving_edges; e; e = e->next_leaving_edge) {
        d = v->distance_from_source + e->positivized_length;
        tv = e->terminal_vertex;
        if(d < tv->distance_from_source) {
          tv->distance_from_source = d;
          tv->pred_edge_toward_source = e;
          if(tv->last_change < iteration) {
            _new_front[new_front_size++] = tv;
            tv->last_change = iteration;
          }
        }
      }
    }

    tmp_front = _new_front; _new_front = _front; _front = tmp_front;

    front_size = new_front_size;

    iteration++;
  } while(front_size > 0);
}

void MTPGraph::find_best_paths(scalar_t *lengths) {
  scalar_t total_length;
  Vertex *v;
  Edge *e;

  for(int e = 0; e < _nb_edges; e++) {
    _edges[e].length = lengths[e];
    _edges[e].occupied = 0;
    _edges[e].positivized_length = _edges[e].length;
  }

  // We call find_shortest_path here to set properly the distances to
  // the source, so that we can make all the edge lengths positive at
  // the first iteration.
  find_shortest_path();

  do {
    update_positivized_lengths();
    force_positivized_lengths();
    find_shortest_path();

    total_length = 0.0;

    // Do we reach the sink?
    if(_sink->pred_edge_toward_source) {
      // If yes, compute the length of the best path according to the
      // original edge lengths
      v = _sink;
      while(v->pred_edge_toward_source) {
        total_length += v->pred_edge_toward_source->length;
        v = v->pred_edge_toward_source->origin_vertex;
      }
      // If that length is negative
      if(total_length < 0.0) {
#ifdef VERBOSE
        cerr << "Found a path of length " << total_length << endl;
#endif
        // Invert all the edges along the best path
        v = _sink;
        while(v->pred_edge_toward_source) {
          e = v->pred_edge_toward_source;
          v = e->origin_vertex;
          e->invert();
          // This is the only place where we change the occupations of
          // edges
          e->occupied = 1 - e->occupied;
        }
      }
    }

  } while(total_length < 0.0);

  // Put back the graph in its original state (i.e. invert edges which
  // have been inverted in the process)
  for(int k = 0; k < _nb_edges; k++) {
    Edge *e = _edges + k;
    if(e->occupied) { e->invert(); }
  }
}

int MTPGraph::retrieve_one_path(Edge *e, Path *path) {
  Edge *f, *next = 0;
  int l = 0;

  if(path) {
    path->nodes[l++] = e->origin_vertex->id;
    path->length = e->length;
  } else l++;

  while(e->terminal_vertex != _sink) {
    if(path) {
      path->nodes[l++] = e->terminal_vertex->id;
      path->length += e->length;
    } else l++;
    int nb_choices = 0;
    for(f = e->terminal_vertex->leaving_edges; f; f = f->next_leaving_edge) {
      if(f->occupied) { nb_choices++; next = f; }
      if(nb_choices == 0) {
        cerr << "retrieve_one_path: Non-sink end point." << endl;
        abort();
      }
      if(nb_choices > 1) {
        cerr << "retrieve_one_path: Non node-disjoint paths." << endl;
        abort();
      }
    }
    e = next;
  }

  if(path) {
    path->nodes[l++] = e->terminal_vertex->id;
    path->length += e->length;
  } else l++;

  return l;
}

void MTPGraph::retrieve_disjoint_paths() {
  Edge *e;

  for(int p = 0; p < nb_paths; p++) delete paths[p];
  delete[] paths;

  nb_paths = 0;
  for(e = _source->leaving_edges; e; e = e->next_leaving_edge) {
    if(e->occupied) { nb_paths++; }
  }

  paths = new Path *[nb_paths];

  int p = 0;
  for(e = _source->leaving_edges; e; e = e->next_leaving_edge) {
    if(e->occupied) {
      int l = retrieve_one_path(e, 0);
      paths[p] = new Path(l);
      retrieve_one_path(e, paths[p]);
      p++;
    }
  }
}
