#ifndef IPCV_IMAGE_INC
#define IPCV_IMAGE_INC

#include "ip_ColorDef.h"
#include "cv.h"
#include "ip_Image.h"

namespace ImageProcessing {

  //#define DEBUG_CLASSES 0

  //-----
  
  /** 
    
      @author Jean-marc Odobez (Jean-Marc.Odobez@idiap.ch)
      @author Daniel Gatica-Perez (gatica@idiap.ch)
  */
  //===================================================
  //
  //    class         ipcv_Image
  //
  //    class for one band image interface using opencv
  //       image representation
  //
  //    Supported types :
  //       - 5 types of ipl librairy (char, unsigned char,
  //          short int, int , float)
  //       - one band of these types, or three bands,
  //         cf ip_ColorElement
  //    
  //    Remarks : The iplImage representation is
  //              left available so that it can be used
  //              with opencv.
  //              When using opencv functions that may
  //              change the image size and storage place,
  //              use the refresh() function to be sure 
  //              that the interface is in concordance
  //              with the current storage room of the 
  //              iplImage structure.
  //
  //===================================================

  //==================================================
  //
  //    function needed to automatically set the 
  //    the type of image element (needed for the
  //    image representation with the opencv image
  //    format).
  //
  //==================================================
  template<class T>
    void  TypeIpl(T a,int & depth,int & nbbands){ 
    printf("Not an IPL type \n"); fflush(stdout); exit(1); return -1;  }
  
  void  TypeIpl(uchar a,int & depth,int & nbbands);
  void  TypeIpl(char a,int & depth,int & nbbands);
  void  TypeIpl(short int a,int & depth,int & nbbands);
  void  TypeIpl(int a,int & depth,int & nbbands);
  void  TypeIpl(float a,int & depth,int & nbbands);
  
  template<class T>
    void  TypeIpl(ip_ColorElement<T>  a,int & depth,int & nbbands){ 
    TypeIpl(a.r,depth,nbbands); nbbands=3;   }

  //===================================================
  //===================================================
  //===================================================

  template <class Type>
    class  ipcv_Image 
    :     public ip_Image<Type> {
    public:
    IplImage    *m_pImage;
    
    // redundant with parameter in image,
    // but speeds up things
    // POTENTIAL DANGER :
    // SOME OPENCV DO REALLOCATION
    // --> DATA AND STEPLINE MAY THUS
    // --> NOT BE "UP-TO-DATE"

    private:
    int         m_iDepth; // to perform reallocation
    int         m_iNbBands;
    char       *m_pData; 
    int         m_iStepLine;
    // for sweeping
    Type       *m_pCurrLine;

    public:

    //----------------
    // constructors
    //
    // 
    ipcv_Image();
    ipcv_Image(int nbli,int nbco);

    //----------------
    // informations members
    //
    // line index : from 0 to nbLines-1
    // col  index : from 0 to nbColumns-1
    
    virtual inline int nbLines(){return m_pImage->height;} 
    virtual inline int nbColumns(){ return m_pImage->width; }
   
    //----------------
    // managing memory and initialisation


    void allocateMem(int nbli,int nbco); // (re) allocate memory, if necessary
    void freeMem();     // destroy memory (should also be done
    // by destructor)

    virtual void init(Type value);  // initialisation to value
    virtual void init(int nbli,int nbco,Type value);  // (re) allocate + initialisation

    // virtual void permute(ip_Image<Type> & tab); // permutation of images (without
    // recopy of content


    //------------------
    virtual ipcv_Image<Type> & affecte(ip_Image<Type> * Ima);
    virtual ipcv_Image<Type> & operator=(ip_Image<Type> & Ima);
    virtual ipcv_Image<Type> & operator=(ipcv_Image<Type> & Ima);

    //------------------
    // accessing elements
    // Note that normally, the () operator and value fonction are faster
    // than the setVal or getVal functions

    // M(i,j)=a; or a=M(i,j);
    virtual inline Type & operator()(int li,int co){
      return *((Type *)(m_pData + m_iStepLine * li) +co);}

    // M.value(i,j)=a; or a=M.value(i,j);
    virtual inline Type & value(int li,int co){
      return *((Type *)(m_pData + m_iStepLine * li) +co);}
    
    virtual inline Type & operator()(int li,int co) const {
      return *((Type *)(m_pData + m_iStepLine * li) +co);}
   
    virtual inline Type & value(int li,int co) const {
      return *((Type *)(m_pData + m_iStepLine * li) +co);}
    
    virtual inline void setVal(int li,int co, Type val){
      *((Type *)(m_pData + m_iStepLine * li) +co)=val;}

    virtual inline Type getVal(int li,int co){
      return *((Type *)(m_pData + m_iStepLine * li) +co);
    }
    
    //------------------
    // sweeping through image
    // Setting image line, and then accessing through
    // column number

    inline void setLine(int li){
      m_pCurrLine=(Type *)(m_pImage->imageData+m_pImage->widthStep*li);
    }
    
    inline Type & valueCol(int co){
      return *(m_pCurrLine+co);
    }
    
    inline Type & valueCol(int co) const {
      return *(m_pCurrLine+co);
    }

    //------------------
    // the refresh function
    inline void refresh(){
      if(m_pImage->nChannels!=m_iNbBands){
	printf("Attention danger \n");
      }
	
      m_iNbBands=m_pImage->nChannels;
      m_pData=m_pImage->imageData;
      m_iStepLine=m_pImage->widthStep;
    }

    //------------------
    virtual ~ipcv_Image(){
      freeMem();
      cvReleaseImageHeader(&m_pImage);
    }
    
  };

  //=======================================================
  //
  //
  //
  //=======================================================

  //----------------
  template <class Type>
  ipcv_Image<Type>::ipcv_Image(){
    CvSize s;

#ifdef  DEBUG_CLASSES
    printf("ipcv_Image : default constructor : 1 \n"); fflush(stdout);
#endif

    TypeIpl((Type)0,m_iDepth,m_iNbBands);
    s.width=0; s.height=0;
    m_pImage=cvCreateImageHeader(s,m_iDepth,m_iNbBands);
    m_pImage->imageData=NULL;
    allocateMem(0,0);
  }
  
  //----------------
  template <class Type>
    ipcv_Image<Type>::ipcv_Image(int nbli,int nbco){

#ifdef  DEBUG_CLASSES
    printf("ipcv_Image : constructor (nbli=%d,nbco=%d)  : 1 \n",nbli,nbco); fflush(stdout);
#endif

    CvSize s;
    TypeIpl((Type)0,m_iDepth,m_iNbBands);
    s.width=nbco; s.height=nbli;
    m_pImage=cvCreateImageHeader(s,m_iDepth,m_iNbBands);
    m_pImage->imageData=NULL;
    allocateMem(nbli,nbco);
  }
  
  
  //----------------
  // managing memory and initialisation
  
  template <class Type>
  void ipcv_Image<Type>::allocateMem(int nbli,int nbco){

    bool alloc=true;

#ifdef  DEBUG_CLASSES
    printf("ipcv_Image : allocateMem  : 1 (nbli=%d nbco=%d)\n",nbli,nbco); fflush(stdout);
#endif
    
    if(m_pImage->imageData!=NULL  && nbli == m_pImage->height && nbco==m_pImage->width){
#if   DEBUG_CLASSES > 2
      printf("ipcv_Image : allocateMem : 2 \n"); fflush(stdout);
#endif
      return;
    }
    
    if(m_pImage->imageData==NULL){
#if   DEBUG_CLASSES > 2
      printf("ipcv_Image : allocateMem : 3 \n"); fflush(stdout);
#endif
      cvInitImageHeader(m_pImage,cvSize(nbco,nbli),m_iDepth,m_iNbBands,
 			IPL_ORIGIN_TL,IPL_ALIGN_DWORD,1);
    }
    else {
      if(nbli != m_pImage->height || nbco != m_pImage->width){ // should be true
#if   DEBUG_CLASSES > 2
	printf("ipcv_Image : allocateMem : 4 \n"); fflush(stdout);
#endif
	// deallocate
	freeMem();
	cvInitImageHeader(m_pImage,cvSize(nbco,nbli),m_iDepth,m_iNbBands,
			  IPL_ORIGIN_TL,IPL_ALIGN_DWORD,1);
	m_pImage->imageData=NULL;
      }
      else{
#if   DEBUG_CLASSES > 2
	printf("ipcv_Image : allocateMem : 5 \n"); fflush(stdout);
#endif
	alloc=false;
      }
      
    }

    
    if(alloc && nbli>0 && nbco>0){
#if   DEBUG_CLASSES > 2
      printf("ipcv_Image : allocateMem  : 6 \n"); fflush(stdout);
#endif
      cvCreateImageData(m_pImage);
      m_pData=m_pImage->imageData;
      m_iStepLine=m_pImage->widthStep;
    }
  }
  
  //----------------
  template <class Type>
  void ipcv_Image<Type>::freeMem(){     // destroy image memory 
#ifdef  DEBUG_CLASSES
    printf("ipcv_Image : freeMem  : 1 \n"); fflush(stdout);
#endif
    cvReleaseImageData(m_pImage);
    m_pImage->imageData=NULL;
    m_pData=NULL;
    m_iStepLine=0;
  }

  //----------------
  template<class Type>
    ipcv_Image<Type> & ipcv_Image<Type>::affecte(ip_Image<Type> * Ima){
    
#ifdef  DEBUG_CLASSES
    printf("ipcv_Image : affecte ip_Image : 1 \n"); fflush(stdout);
#endif
    
    if(this != Ima){
      int li,co;
#if   DEBUG_CLASSES > 2
      printf("ipcv_Image : affecte  : 2 \n"); fflush(stdout);
#endif
      (*this).allocateMem(Ima->nbLines(),Ima->nbColumns());
      for(li=0;li<nbLines();li++){
	Ima->setLine(li);
	setLine(li);
	for(co=0;co<nbColumns();co++)
	  valueCol(co)=Ima->valueCol(co);
      }
    }
    return *this;
  }
  
  //----------------
  template<class Type>
    ipcv_Image<Type> & ipcv_Image<Type>::operator=(ip_Image<Type> & Ima){
    
#ifdef  DEBUG_CLASSES
    printf("ipcv_Image : operator= ip_Image : 1 \n"); fflush(stdout);
#endif
    
    affecte(&Ima);
    return *this;
  }

  //----------------
  template<class Type>
    ipcv_Image<Type> & ipcv_Image<Type>::operator=(ipcv_Image<Type> & Ima){
    
#ifdef  DEBUG_CLASSES
    printf("ipcv_Image : operator= ipcv_Image : 1 \n"); fflush(stdout);
#endif

    affecte(&Ima);
    return *this;
  }

  //----------------
  template<class Type>
  void ipcv_Image<Type>::init(Type val){  // initialisation to value
    int li,co;
    for(li=0;li<nbLines();li++){
      setLine(li);
      for(co=0;co<nbColumns();co++)
	valueCol(co)=val;
    }
  }

  //----------------
  template<class Type>
  void ipcv_Image<Type>::init(int nbli,int nbco,Type val){  // (re) allocate + initialisation
    allocateMem(nbli,nbco);
    init(val);
  }




  //=======================================================
  //
  //
  //
  //=======================================================

  template <class Type>
    class ipcv_ImageAllocator :
    public  ip_ImageAllocator<Type> {

    public:
    virtual ip_Image<Type> * newIpImage(){  
      return new ipcv_Image<Type> (0,0);
    }

  };


   
}

#endif
