// uses open cv
#include "ipcv_HistogramTemplate.h"


namespace ImageProcessing {

#define  Val3(ad,i1,i2,i3,n)  *(ad+(n*i1+i2)*n+i3)


  ipcv_HistogramTemplate::ipcv_HistogramTemplate(){
    m_pHt=NULL;
    m_iNbBands=0;      // Number of bands in the images
    m_iNbPositions=0;  // Number of positions/boundingboxes
    m_iDistanceType=HIST_DISTANCE_BHATACHARYYA;
  };

  //-----
  ipcv_HistogramTemplate::ipcv_HistogramTemplate(int _NbPos,int _NbDims,
						 int *_Dims,float **ranges,
						 bool _unif){
    m_pHt=NULL;
    createHistogram(_NbPos,_NbDims,_Dims,ranges,_unif);
  }

  //----
  ipcv_HistogramTemplate::ipcv_HistogramTemplate(int _NbPos,int _NbDims,int _NBinPerBand,
						 float _minRange,float _maxRange){
    m_pHt=NULL;
    createHistogramUniform(_NbPos,_NbDims,_NBinPerBand,_minRange,_maxRange);
  }

  //-----
  // Number of bins per dimension; if not specified
  // total number of bins (per position)
  int   ipcv_HistogramTemplate::nbBins(int k){
    if(k>=0 && k<m_iNbBands)
      return m_pHt[0]->dims[k];
    else {
      int     i,Nbin=1;
      for(i=0;i<m_iNbBands;i++)
	Nbin *= m_pHt[0]->dims[i];
      return Nbin;
    }
  }

  //-----
  void ipcv_HistogramTemplate::deleteHistogram(){
    int i;
    if(m_pHt!=NULL){
      for(i=0;i<m_iNbPositions;i++){
	cvReleaseHist(&m_pHt[i]);
      }
      delete [] m_pHt;
    }
  }  
  
  //-----
  void ipcv_HistogramTemplate::createHistogram(int _NbPos,int _NbDims,
						       int *_Dims,float **ranges,
						       bool _unif){
    int i,uni=0;
    if(_unif) uni=1;
    if(m_pHt!=NULL){
      deleteHistogram();
    }
    m_iNbBands=_NbDims;
    m_iNbPositions=_NbPos;
    m_pHt=new CvHistogram * [_NbPos];
    for(i=0;i<_NbPos;i++){
      m_pHt[i] = cvCreateHist(_NbDims,_Dims,CV_HIST_ARRAY,0,uni);  
      cvSetHistBinRanges(m_pHt[i],ranges,uni);
    }
    m_iDistanceType=HIST_DISTANCE_BHATACHARYYA;
    
  }
    //-----
  void ipcv_HistogramTemplate::createHistogramUniform(int _NbPos,int _NbDims,int _NbBin,
						      float _minRange,float _maxRange){
    float eps=0.0001;
    int i,*dims;
    float **ranges;
    
    dims=new int[_NbDims];
    ranges = new float * [_NbDims];
    for(i=0;i<_NbDims;i++){
      ranges[i]=new float [2];
      ranges[i][0]=_minRange;ranges[i][1]=_maxRange+eps;
      dims[i]=_NbBin;
    }

    createHistogram(_NbPos,_NbDims,dims,ranges,true);

    for(i=0;i<_NbDims;i++){
      delete [] ranges[i];
    }
    delete [] ranges;
    delete [] dims;
  }

  // create a copy with same characteristics
  //-----
  ipcv_HistogramTemplate * ipcv_HistogramTemplate::createCopy(bool InitZero){

    ipcv_HistogramTemplate *H;
    int i;

    H= new ipcv_HistogramTemplate();
    H->m_iNbPositions=m_iNbPositions;
    H->m_pHt=new CvHistogram * [H->m_iNbPositions];
    for(i=0;i<H->m_iNbPositions;i++){
      H->m_pHt[i]=NULL;
      cvCopyHist(m_pHt[i],&H->m_pHt[i]);
		// create off course uninitialized read
		//
      // if(InitZero)
		//    cvNormalizeHist(H->m_pHt[i],0.);
    }
    H->m_iNbBands=m_iNbBands;
    H->m_iDistanceType=m_iDistanceType;
    return H;
  }


  //-----
  void ipcv_HistogramTemplate::setDistanceType(ipcv_HistDistance _dist_type){
    m_iDistanceType=_dist_type;
  }
   
  //-----
  float ipcv_HistogramTemplate::distance(ipcv_HistogramTemplate  *H){
    int  i;
    double d=0.;

    if(!sameTemplate(H)){
      printf("\nDistance between two histograms without the same characteristics\n");
      exit(0);
    }
    
    //
    switch(m_iDistanceType){
    case HIST_DISTANCE_CHISQR:
    case HIST_DISTANCE_INTERSECT:
      d=0.;
      for(i=0;i<m_iNbPositions;i++)
		  d+=cvCompareHist(H->m_pHt[i],m_pHt[i],(CvCompareMethod)m_iDistanceType);
      if(m_iDistanceType==HIST_DISTANCE_INTERSECT)
		  d=1.-d;
      break;

    case HIST_DISTANCE_BHATACHARYYA:
      d=0.;
      if(m_pHt[0]->type==CV_HIST_ARRAY) {
		  float  *pbin1,*pbin2,*pend;
		  
		  for(i=0;i<m_iNbPositions;i++){
			 pbin1=H->m_pHt[i]->array;
			 pbin2=m_pHt[i]->array;
			 pend=pbin1+nbBins();

			 //printf("\n");
			 while(pbin1<pend){
				// remarks : cvSqrt not as precise than sqrt
				// printf("%7.4f  %7.4f\n",*pbin1,*pbin2);
				d+=cvSqrt(*pbin1++* *pbin2++);
			 }
		  }
		  d=cvSqrt(1.-d);
      }
      else {
		  printf("Warning : bhatacharrya coef computation not implemented with trees\n\n");
		  exit(0);
      }
      break;

	 case HIST_DISTANCE_HAUSSDORF:
		{
      d=0.;
		bool processd=true;

		if(nbDimensions()!=3){
		  printf("Haussdorf distance not implemented with this number of dimensions\n");
		  fflush(stdout);
		  processd=false;
		}
		
		for(i=1;i<nbDimensions();i++)
		  if(nbBins(i)!=nbBins(0)){
			 printf("Nb bins assumed to be the same along each dimension\n");
			 fflush(stdout);			 
			 processd=false;
		  }
		if(processd==false)
		  exit(0);
		
		
      if(m_pHt[0]->type==CV_HIST_ARRAY) {
		  float  *pbin1,*pbin2,v,dv,dvc;
		  int    p,j,k,ii,jj,kk,N=nbBins(0);
		  
		  
		  for(p=0;p<m_iNbPositions;p++){
			 pbin1=H->m_pHt[p]->array;
			 pbin2=m_pHt[p]->array;
			 
			 for(i=0;i<N;i++){
				for(j=0;j<N;j++){
				  for(k=0;k<N;k++){
					 v=Val3(pbin1,i,j,k,N); dv=1.e18;
					 // normally dv=1. should suffice as initialization
					 // since histograms are supposes to be normalized
					 for(ii=i-1;ii<=i+1;ii++){
						if(ii>=0 && ii<N){
						  for(jj=j-1;jj<=j+1;jj++){
							 if(jj>=0 && jj<N){
								for(kk=k-1;kk<=k+1;kk++){
								  if(kk>=0 && kk<N){
									 dvc=fabs(v-Val3(pbin2,ii,jj,kk,N));
									 if(dvc<dv)
										dv=dvc;
								  }
								}
							 }
						  }
						}
					 }// end of looking for the minimum
					 d=d+dv;
				  }
				}
			 }
			 // sum has been performed for all the bin
		  }// for all positions
		}
      else {
		  printf("Warning : Haussdorf distance not implemented with trees\n\n");
		  exit(0);
      }
		}
		
      break;
      
    default:
      printf("Warning : no such histogram distance defined\n\n");
      exit(0);
    }

    return (float)d;

  };

  //-----
  bool  ipcv_HistogramTemplate::sameTemplate(ipcv_HistogramTemplate  *H){
    bool same=true;
    int i;
    if(H->m_iNbPositions!=m_iNbPositions) same=false;
    if(H->m_iNbBands!=m_iNbBands) same=false;
    for(i=0;i<m_iNbBands;i++)
      if(nbBins(i)!=H->nbBins(i))
	same = false;
    return same;
  }
  
  //-----
  void ipcv_HistogramTemplate::display(char *comments){
	 printf("Histogram template : %s\n",comments);
	 printf("Nb Bands = %d - Nb Positions = %d  - DistanceType = %d \n",
			  m_iNbBands,m_iNbPositions,m_iDistanceType);
	 fflush(stdout);

	 if(m_pHt[0]->type==CV_HIST_ARRAY) {
		float  *pbin1,*pend;
		for(int i=0;i<m_iNbPositions;i++){
		  pbin1=m_pHt[i]->array;
		  pend=pbin1+nbBins();
		  printf("Position %d\n",i);
		  while(pbin1<pend){
			 printf("%5.2f",*pbin1++);
		  }
		  printf("\n");
		}
	 }
	 printf("\n");
	 
  }
  

  //-----
  void ipcv_HistogramTemplate::load(char *_name){}

  //-----
  void ipcv_HistogramTemplate::save(char *_name){}
  //-----


}

