#include "general.h"
#include "ip_Image.h"
#include "ip_ColorDef.h"
#include "ip_Shape.h"
#include "ip_PixelAndPoint.h"
#include "ip_DrawFunctions.h"
#include "ip_Ellipse.h"


namespace ImageProcessing {

  //----- 
  ip_Ellipse::ip_Ellipse()
  {
    m_pCentroid = new ip_Point[1];
    m_pSemiAxis = new ip_Point[1];
    m_pExtremePoints = new ip_Point[4];
    m_pCentroid->m_rX = 0;
    m_pCentroid->m_rY = 0;
    m_pSemiAxis->m_rX = 0;
    m_pSemiAxis->m_rY = 0;
    m_dAngle = 0;
  }
  
  //----
  ip_Ellipse::ip_Ellipse(real centroidX, real centroidY, real semiAxisX,
			 real semiAxisY, real angle)
  {
    m_pCentroid = new ip_Point[1];
    m_pSemiAxis = new ip_Point[1];
    m_pExtremePoints = new ip_Point[4];
    init(centroidX, centroidY, semiAxisX, semiAxisY, angle);
  }

  //----
  ip_Ellipse::ip_Ellipse(real *pPars)
  {    
    m_pCentroid = new ip_Point[1];
    m_pSemiAxis = new ip_Point[1];
    m_pExtremePoints = new ip_Point[4];
    init(pPars);
  }

  //-----
  // copy constructor
  ip_Ellipse::ip_Ellipse(const ip_Ellipse & ellipse)
  {
	 int i;
    m_pCentroid[0]=ellipse.m_pCentroid[0];
	 m_pSemiAxis[0]=ellipse.m_pSemiAxis[0];
	 for(i=0;i<4;i++)
		m_pExtremePoints[i]=ellipse.m_pExtremePoints[i];
    m_dAngle = ellipse.m_dAngle;
  }

  //----
  void ip_Ellipse::init(real centroidX, real centroidY, real semiAxisX,
			real semiAxisY, real angle)
  {
    m_pCentroid->m_rX = centroidX;
    m_pCentroid->m_rY = centroidY;
    m_pSemiAxis->m_rX = semiAxisX;
    m_pSemiAxis->m_rY = semiAxisY;
    m_dAngle = angle;
    getExtremePointsFromPars();
  }

  //---- assuming a specific implementation 
  void ip_Ellipse::init(real *pPars)
  {
    m_pCentroid->m_rX = pPars[0];
    m_pCentroid->m_rY = pPars[1];
    m_pSemiAxis->m_rX = pPars[2];
    m_pSemiAxis->m_rY = pPars[3];
    m_dAngle = pPars[4];
    getExtremePointsFromPars();
  }

  //---- 
  void ip_Ellipse::init(ip_Ellipse *pE)
  {
    m_pCentroid->m_rX = pE->m_pCentroid->m_rX;
    m_pCentroid->m_rY = pE->m_pCentroid->m_rY;
    m_pSemiAxis->m_rX = pE->m_pSemiAxis->m_rX;
    m_pSemiAxis->m_rY = pE->m_pSemiAxis->m_rY;
    m_dAngle          = pE->m_dAngle;
    getExtremePointsFromPars();
  }

  //-----
  ip_Ellipse & ip_Ellipse::operator=(const ip_Ellipse & ellipse)
  {
	 if(this != &ellipse){
		int i;
		m_pCentroid[0]=ellipse.m_pCentroid[0];
		m_pSemiAxis[0]=ellipse.m_pSemiAxis[0];
		for(i=0;i<4;i++)
		  m_pExtremePoints[i]=ellipse.m_pExtremePoints[i];
		m_dAngle = ellipse.m_dAngle;
	 }
	 return *this;
	 
  }

  //----
  void ip_Ellipse::getExtremePointsFromPars(void)
  {
    real tempX, tempY;
    real stepSize = 6.28318/4;

    for(int i=0;i<4;i++)
    {
      tempX = m_pSemiAxis->m_rX * cos(stepSize*i);
      tempY = m_pSemiAxis->m_rY * sin(stepSize*i);
      m_pExtremePoints[i].m_rX = m_pCentroid->m_rX + cos(m_dAngle)*tempX - 
	sin(m_dAngle)*tempY;
      m_pExtremePoints[i].m_rY = m_pCentroid->m_rY + sin(m_dAngle)*tempX + 
	cos(m_dAngle)*tempY;
    }
  }

  //----
  void ip_Ellipse::getParsFromExtremePoints(void)
  {
    m_pCentroid->m_rX = (m_pExtremePoints[0].m_rX + m_pExtremePoints[2].m_rX)/2.;
    m_pCentroid->m_rY = (m_pExtremePoints[0].m_rY + m_pExtremePoints[2].m_rY)/2.;
    m_pSemiAxis->m_rX = ip_euclideanDistance(m_pExtremePoints[0],m_pExtremePoints[2])/2.;
    m_pSemiAxis->m_rY = ip_euclideanDistance(m_pExtremePoints[1],m_pExtremePoints[3])/2.;
    m_dAngle          = ip_angle(m_pExtremePoints[0],m_pExtremePoints[2]); 
  }
 

  //----- 
  //      sample from an arbitrary ellipse as follows:
  //      1: sample points from the "centered" ellipse using polar representation
  //         (this assumes that scale is already correct)
  //      2: apply rotation and translation to each point
  //-----

  void ip_Ellipse::samplePoints(ip_Point *pPointList, int NumberOfPoints)
  {    
    real tempX, tempY;
    real stepSize = 6.28318/NumberOfPoints;

    for(int i=0;i<NumberOfPoints;i++)
    {
      tempX = m_pSemiAxis->m_rX * cos(stepSize*i);
      tempY = m_pSemiAxis->m_rY * sin(stepSize*i);

      pPointList[i].m_rX = m_pCentroid->m_rX + cos(m_dAngle)*tempX - 
	sin(m_dAngle)*tempY;
      pPointList[i].m_rY = m_pCentroid->m_rY + sin(m_dAngle)*tempX + 
	cos(m_dAngle)*tempY;
    }
  } 


  //----- assumes that memory exists, normal vector points outside curve
  void ip_Ellipse::computeNormalToPoint(ip_Point *pPoint, ip_Point *pNormal)
  {
    pNormal->m_rX = (pPoint->m_rX - m_pCentroid->m_rX);
    pNormal->m_rY = (pPoint->m_rY - m_pCentroid->m_rY);
    real l = sqrt(pNormal->m_rX*pNormal->m_rX + pNormal->m_rY*pNormal->m_rY);
    pNormal->m_rX = pNormal->m_rX / l;
    pNormal->m_rY = pNormal->m_rY / l;
  } 


  //----- assumes that memory exists, normal vector points outside curve
  void ip_Ellipse::computeNormalToPoint(ip_Point *pNormal, real angle)
  {
    real den;
    real tempX, tempY, nX, nY;    

    nY = tempX = m_pSemiAxis->m_rX * sin(angle);
    nX = tempY = m_pSemiAxis->m_rY * cos(angle);
    den   = sqrt(tempX*tempX + tempY*tempY);
    nX = nX/den;
    nY = nY/den;

    pNormal->m_rX = cos(m_dAngle)*nX - sin(m_dAngle)*nY;
    pNormal->m_rY = sin(m_dAngle)*nX + cos(m_dAngle)*nY;   
  } 


  //----- 
  void ip_Ellipse::computeNormalVectors(ip_Point *pPointList, 
					ip_Point *pNormalList,
					int NumberOfPoints)
  {
    for(int i=0;i<NumberOfPoints;i++)
      computeNormalToPoint(pPointList+i, pNormalList+i);
  } 


  //----- assumes that memory exists, normal vector points outside curve
  void ip_Ellipse::computeNormalVectors(ip_Point *pNormalList, int NumberOfPoints)
  {        
    real stepSize = 6.28318/NumberOfPoints;
    for(int i=0;i<NumberOfPoints;i++)
      computeNormalToPoint(pNormalList+i, stepSize*i);
  }

  //-----     
  void ip_Ellipse::translate(int dep_co,int dep_li)
  {
    m_pCentroid->m_rX += dep_co;
    m_pCentroid->m_rY += dep_li;
    getExtremePointsFromPars();
  }

  //-----     
  void ip_Ellipse::translateScale(int dep_co,int dep_li, real scale)
  {
    m_pCentroid->m_rX += dep_co;
    m_pCentroid->m_rY += dep_li;
    m_pSemiAxis->m_rX *= scale;
    m_pSemiAxis->m_rY *= scale;
    getExtremePointsFromPars();
  }


  //-----
  void  ip_Ellipse::draw(ip_Point *pPointList, 
			 int NumberOfPoints,
			 ip_Image<ip_ColorElement8u> & Image,
			 ip_ColorElement8u color)
   {
     ip_drawClosedListOfPoints(NumberOfPoints, pPointList, 
			       Image, color);
   }


  //-----
  void  ip_Ellipse::draw(ip_Point *pPointList, 			 
			 int NumberOfPoints,
			 ip_Image<ip_ColorElement8u> *pImage,
			 ip_ColorElement8u color)
   {
	  int i;
	  
     ip_drawClosedListOfPoints(NumberOfPoints, pPointList, pImage, color);

	  // to augmente ellipse thickness
	  for(i=0;i<NumberOfPoints;i++){
		 pPointList[i].m_rX+=1.;
	  }
     ip_drawClosedListOfPoints(NumberOfPoints, pPointList, pImage, color);
	  for(i=0;i<NumberOfPoints;i++)
		 pPointList[i].m_rX-=2.;
     ip_drawClosedListOfPoints(NumberOfPoints, pPointList, pImage, color);

	  for(i=0;i<NumberOfPoints;i++){
		 pPointList[i].m_rY+=1.;
		 pPointList[i].m_rX+=1.;
	  }
     ip_drawClosedListOfPoints(NumberOfPoints, pPointList, pImage, color);
	  for(i=0;i<NumberOfPoints;i++)
		 pPointList[i].m_rY-=2.;
     ip_drawClosedListOfPoints(NumberOfPoints, pPointList, pImage, color);
	  

   }


  //-----
  bool ip_Ellipse::insideEnough(int rows, int cols)
   {
     // rule: if any three extreme points are inside the image, the measurement is
     // considered valid: this amounts to at least 50% of the shape inside

     if( (ip_inImage(m_pExtremePoints[0],rows, cols) && ip_inImage(m_pExtremePoints[1],rows,cols) 
	  && ip_inImage(m_pExtremePoints[2],rows,cols))
	 || (ip_inImage(m_pExtremePoints[0],rows, cols) && ip_inImage(m_pExtremePoints[1],rows, cols) 
	     && ip_inImage(m_pExtremePoints[3],rows, cols))
	 || (ip_inImage(m_pExtremePoints[0],rows, cols) && ip_inImage(m_pExtremePoints[2],rows, cols) 
	     && ip_inImage(m_pExtremePoints[3],rows, cols))
	 || (ip_inImage(m_pExtremePoints[1],rows, cols) && ip_inImage(m_pExtremePoints[2],rows, cols) 
	     && ip_inImage(m_pExtremePoints[3],rows, cols)) )
       return true;
     else
       return false;
   }


  //----
  void ip_Ellipse::displayPars(void)
  {
    printf("Centroid: (%f,%f); SemiAxis: (%f,%f); Angle: %f\n", 
	   m_pCentroid->m_rX, m_pCentroid->m_rY, 
	   m_pSemiAxis->m_rX,m_pSemiAxis->m_rY, m_dAngle);       
    for(int i=0;i<4;i++)
      printf("Point %hd:(%f,%f)\n", i,m_pExtremePoints[i].m_rX,  m_pExtremePoints[i].m_rY);
  }

 
  //----
  void ip_Ellipse::freeMemory(void)
  {
    if(m_pCentroid != NULL)
    {
      delete[] m_pCentroid;
      m_pCentroid = NULL;
    }
    if(m_pSemiAxis != NULL)
    {
      delete[] m_pSemiAxis;
      m_pSemiAxis = NULL;
    }
    if(m_pExtremePoints != NULL)
    {
      delete[] m_pExtremePoints;
      m_pExtremePoints = NULL;
    }
  }


}









