#ifndef IP_BOUNDING_BOX_HH
#define IP_BOUNDING_BOX_HH


#include <stdlib.h>
#include <stdio.h>
#include <iostream.h>
#include <iomanip.h>

#include "ip_Image.h"

#include "general.h"

#ifndef min
/// The min function
#define	min(a,b) ((a) > (b) ? (b) : (a))
#endif

#ifndef max
/// The max function
#define	max(a,b) ((a) > (b) ? (a) : (b))
#endif


namespace ImageProcessing  {

 /** 
     Basic class interface for representing bounding boxes.
      
    
      @author Jean-marc Odobez (Jean-Marc.Odobez@idiap.ch)
      @author Daniel Gatica-Perez (gatica@idiap.ch)
  */
  
  /*

    Bounding box   : 
         - simply defines the limit of a window in an image.
         - line numbers increasing from top to bottom
	 - column number increasing from left to right
	 NOTE :
	   - last line and last column are NOT included in bounding box
	   - we keep the same "axis representations" as with 
             pixels and points ;
	     That is, we always start by providing 
             X or COLUMN or WIDTH before Y or LINE or HEIGHT

	 * constructors :

	 - ip_BoundingBox   F()

	 - ip_BoundingBox   F(first_col,first_line,
	                      last_col,last_line)
            ATTENTION : LAST LINES AND LAST COLUMNS NOT INCLUDED

         - ip_BoundingBox   F(nb_columns,nb_lines)
           First lines and columns assumed to be 0

	 * Functions :

	 - init(first_col,first_line,
	                      last_col,last_line)
                ATTENTION : LAST LINES AND LAST COLUMNS NOT INCLUDED

	 - initPosAndSize(first_col,first_line,
	                  width,height);
               

         - initSize(width,height)

         - firstLine(),firstColumn(),lastLine(),lastColumn()
                ATTENTION : LAST LINES AND LAST COLUMNS NOT INCLUDED

	 - width(), height()

	 - translate(dep_co,dep_li)

	 - display(comment) 

	 - inside(column,line)

	 - empty()

	 - area()

	 - intersection(Bbox)

	 - outerBoundingBox(Bbox)

	 - draw(ip_Image<T> & Ima,T color,int Thickness=1){

 */


  class  ip_BoundingBox
    {
    public: // members : same as opencv, so that we can do casting
      int m_iFirstColumn; // named x in opencv 
      int m_iFirstLine; // named y in opencv
      int m_iWidth;
      int m_iHeight;
      

      //----
      ip_BoundingBox(){
	init(0,0,0,0);
      }

      //----
      ip_BoundingBox(int first_col,int first_line,
			     int last_col,int last_line){
	init(first_col,first_line,last_col,last_line);
      }

      //            ATTENTION : LAST LINES AND LAST COLUMNS NOT INCLUDED

      //----
      //     First lines and columns assumed to be 0
      ip_BoundingBox(int nb_columns,int nb_lines){
	init(0,0,nb_columns,nb_lines);
      }

      //----
      // Functions :
      //----
      // ATTENTION : LAST LINES AND LAST COLUMNS NOT INCLUDED
      inline virtual void init(int first_col,int first_line,int last_col,int last_line){
	m_iFirstLine=first_line; m_iFirstColumn=first_col;
	m_iHeight=last_line-first_line;
	m_iWidth=last_col-first_col;
	m_iWidth=max(m_iWidth,0);
	m_iHeight=max(m_iHeight,0);
      }


      inline virtual void initPosAndSize(int first_col,int first_line,
					 int width,int height){
	m_iFirstLine=first_line; m_iFirstColumn=first_col;
	m_iWidth=width;  m_iHeight=height;
	m_iWidth=max(m_iWidth,0);
	m_iHeight=max(m_iHeight,0);
      }
               
      inline virtual void initSize(int width,int height){
	m_iWidth=width;  m_iHeight=height;
	m_iWidth=max(m_iWidth,0);
	m_iHeight=max(m_iHeight,0);
      }

      //----
      inline virtual int  firstLine(){ return m_iFirstLine;}
      inline virtual int  firstColumn(){ return m_iFirstColumn; }
      //  ATTENTION : LAST LINES AND LAST COLUMNS NOT INCLUDED
      inline virtual int  lastLine(){ return m_iFirstLine+m_iHeight;}
      inline virtual int  lastColumn(){ return m_iFirstColumn+m_iWidth; }

      //-------
      inline virtual int  width(){ return m_iWidth;}
      inline virtual int  height(){ return m_iHeight; }

      //-------
      inline virtual void translate(int dep_co,int dep_li){
	m_iFirstLine+=dep_li;
	m_iFirstColumn+=dep_co;
      }

      //-------
      virtual void  display(char *comment="",FILE *f=stdout){
	printf("\nBounding box : %s \n",comment);
	printf("  First line : %5d  First column : %5d  -- Height : %5d Width : %5d\n",
	       m_iFirstLine,m_iFirstColumn,m_iHeight,m_iWidth);
	printf("  Last  line : %5d  Last  column : %5d \n",
	       lastLine(),lastColumn());
	fflush(f);
      }
      
      //-------
      inline virtual bool inside(int column,int line){
	if(line>=m_iFirstLine && column>=m_iFirstColumn && line < lastLine() && column < lastColumn())
	  return true;
	else
	  return false;
      }

      //-------
      inline virtual int area(){
	return height()*width();
      }
      
      //-------
      inline virtual bool empty(){
	if(area()<=0)
	  return true;
	else
	  return false;
      }

      //-------
      inline virtual void intersection(ip_BoundingBox & BBox2){
	ip_BoundingBox B(
			 max(firstColumn(),BBox2.firstColumn()),
			 max(firstLine(),BBox2.firstLine()),
			 min(lastColumn(),BBox2.lastColumn()),
			 min(lastLine(),BBox2.lastLine())
			 );
	(*this)=B;
      }


      //-------
      inline virtual void outerBoundingBox(ip_BoundingBox & BBox2){
	ip_BoundingBox B(min(firstColumn(),BBox2.firstColumn()),
			 min(firstLine(),BBox2.firstLine()),
			 max(lastColumn(),BBox2.lastColumn()),
			 max(lastLine(),BBox2.lastLine())
			 );
	(*this)=B;
      }

      //-------
      template <class T>
	inline void draw(ip_Image<T> & Ima,
			 T color,int Thickness=1){
	ip_BoundingBox C(Ima.nbColumns(),Ima.nbLines());
	int li,co;

	if(Thickness>1){
	  C.initPosAndSize(1,1,Ima.nbColumns()-2,Ima.nbLines()-2);
	}
	C.intersection(*this);

	if(C.area()>0){
	  co=C.lastColumn()-1;
	  for(li=C.firstLine();li<C.lastLine();li++){
	    Ima(li,C.firstColumn())=color;
	    Ima(li,co)=color;
	  }
	  li=C.lastLine()-1;
	  for(co=C.firstColumn();co<C.lastColumn();co++){
	    Ima(C.firstLine(),co)=color;
	    Ima(li,co)=color;
	  }
	  if(Thickness>1){
	    int li2,co2=C.lastColumn();
	    co=C.lastColumn()-2;
	    for(li=C.firstLine()-1;li<C.lastLine()+1;li++){
	      Ima(li,C.firstColumn()+1)=color;
	      Ima(li,co)=color;
	      Ima(li,C.firstColumn()-1)=color;
	      Ima(li,co2)=color;
	    }
	    li2=C.lastLine();
	    li=C.lastLine()-2;
	    for(co=C.firstColumn()-1;co<C.lastColumn()+1;co++){
	      Ima(C.firstLine()+1,co)=color;
	      Ima(li,co)=color;
	      Ima(C.firstLine()-1,co)=color;
	      Ima(li2,co)=color;
	    }
	  }
	}
      }
	
    };
} 



#endif  // IP_BOUNDING_BOX
