#ifndef BICV_BOUNDING_BOXES_TEMPLATE_INC
#define BICV_BOUNDING_BOXES_TEMPLATE_INC

#include "general.h"
#include "bf_RandomVariable.h"
#include "bf_RandomGenerator.h"
#include "bf_DistributionReal.h"

#include "bf_RVFunctions.h"

// open cv
#include "cv.h"

#include "ipcv_HistogramTemplate.h"
#include "bi_HistogramMeasurer.h"
#include "ip_BoundingBox.h"
#include "ip_PlanarTransform.h"

using namespace ImageProcessing;

namespace Torch {
  //-----

  /** 

      
    
      @author Jean-marc Odobez (Jean-Marc.Odobez@idiap.ch)
      @author Daniel Gatica-Perez (gatica@idiap.ch)
  */
  //===================================================
  //
  //  Object defined by a set of bounding boxes.
  // 
  //  Transformation are planar transformation
  //
  //  Transform  of a bounding box b = bounding box that 
  //              includes the 4 transform of the cornes of b.
  //
  //  Parameters of the configuration : 
  //      - should be provided inside a vector of reals.
  //        This vector may contain other parameters not related
  //         to the configuration. Nevertheless, the "useful"
  //         parameters (i.e. the ones that define the configuration) 
  //         should be placed regularly in this vector, the first one
  //         being at the 0 position.
  //      - when providing the configuration parameters (cf setTrasnformReal)
  //        the "step size" between two useful parameters in the input 
  //        vector can be set with the setStepSet() function.
  //      - when recovering configuration parameters (getTransformReal)
  //        the "step size" between two places to store two consecutive parameters
  //        can be specified with the setStepGet() function.
  //
  //===================================================

  class   bicv_BoundingBoxesTemplate : virtual public bi_HistogramMeasurer {
   public:
    ip_BoundingBox          *m_pBBTemplate;
    ip_BoundingBox          *m_pBBCurrent;
    ip_PlanarTransform      *m_pTransform;

  private:
    bool                     m_bTemplateAllocated;
    int                      m_iNbBBoxes;
    int                     *m_pVolumeOfBoxes;
    int                      m_iStepGet;

    // position of the center of the set of Bounding Boxes
    // (whenever it is computed)
    ip_Pixel                 m_cCenter; 
    
  public:


    //-----
    bicv_BoundingBoxesTemplate();

    //-----
    bicv_BoundingBoxesTemplate(int NbBBoxes,ip_PlanarTransform *Transf,
			       bool AllocateMemTemp=false);

    //-----
    bicv_BoundingBoxesTemplate(ip_BoundingBox   *BBTemplate,
			       int NbBBoxes,ip_PlanarTransform *Transf,
			       bool AllocateMemTemp=false,
			       bool with_center=false);


    //----
    virtual void  create(ip_BoundingBox   *BBTemplate,
			 ip_PlanarTransform *Transf,
			 int                 NbBBoxes,
			 bool AllocateMemTemp=false,
			 bool with_center=false);

    //--------------------
    //----
    inline int    nbBBoxes(){ return m_iNbBBoxes;}
    //----

    inline int *   getPVolumeOfBoxes()
      {	return m_pVolumeOfBoxes;   }
    

    //--------------------
    virtual void setStepSet(int step) { m_pTransform->m_iStepRead=step;}
 
    //--------------------
    virtual void setStepGet(int step) { m_iStepGet=step; }

    //--------------------
    //<<<<<<< bicv_BoundingBoxesTemplate.h
    // compute the center of gravity of the template bounding box
    //virtual ip_Pixel computeCenter();
    //=======
    // compute the center of gravity of the current bouding boxes
    virtual ip_Pixel   computeCenter();
    //>>>>>>> 1.3

    //---------------------
    // retrieve the center of gravity of template BB, used last call to center() 
    virtual ip_Pixel getCenter();

    //---------------------
    // center the template BB around the center of gravity
    virtual void center();

    //---------------------
    //-- set the bounding box template with initialisation
    virtual void   setBB(ip_BoundingBox   *BBTemplate,bool with_center=true);

    // MINIMUM FUNCTIONS TO BE DEFINED :

    virtual bool isReal() { return true;}
    virtual int nbParam(){ 
      if(m_pTransform!=NULL) return m_pTransform->m_iNbParams; 
      else return 0;
    }

    //----
    virtual ip_BoundingBox  transformBB(ip_BoundingBox & BBin);

    //----
    // from bi_ShapeTransformTemplate
    virtual void transform(void);
    
    //-----
    // from bi_ShapeTransformTemplate
    virtual inline void getTransformReal(real *pTrPars){
      m_pTransform->getParams(pTrPars,m_iStepGet);
    }

    //----- dgp
    // inherited from bi_ShapeTransformTemplate
    virtual inline real *getTransformReal(void)
    {return m_pTransform->getPtrToParams();}

    //-----
    // from bi_ShapeTransformTemplate
    virtual inline void setTransformReal(real *pTrPars){
      m_pTransform->setParams(pTrPars);
    }

    //----
    // from HistogramMeasurer
    virtual bool  computeHistogram(IplImage **bands,
				   ipcv_HistogramTemplate *H);


    // ----
    virtual void   draw(ip_Image<ip_ColorElement8u> &Ima,ip_ColorElement8u color,
			bool current=true);
    
     // ----
    virtual void   draw(ip_Image<uchar> &Ima,uchar color,
			bool current=true);
    
    //----
    // private:
    virtual void  freeMemory();

    //----
  public:
    virtual ~bicv_BoundingBoxesTemplate(){
      freeMemory();
    }

  };

   
}

#endif
