#ifndef BF_DYNAMIC_AR_INC
#define BF_DYNAMIC_AR_INC

#include "general.h"
#include "Distribution.h"
#include "bf_PredictModel.h"

namespace Torch {

  //-----
  /*
       Some  specific  AR dynamical  model (for real essentially)

  */
  //===================================================
  //
  //  class  bf_DynamicAR
  //
  //  different linear AR model
  //  
  //
  //  Model Dynamic_AR2 :
  //    [ x_k; x_{k-1} ] = [ 2 , -1 ; 1 , 0][x_{k-1}; x_{k-2}] + [ 1; 0 ] white noise
  //
  //  Model Dynamic_AR2bis :
  //    [ x_k; x_{k-1}; x_{k-2} ] = [3,-3,1;1,0,0;0,1,0][x_{k-1}; x_{k-2};x_{k-3}] 
  //                               + [1;0;0] white noise
  //
  //===================================================

  typedef enum {
    DynamicAR2,
    DynamicAR2bis,
    DynamicAR3bis,
    DynamicAR3
  } bf_DynamicARModel;

  //===================================================
  class bf_DynamicAR : public bf_SamplePredictModelReal  {
  public:
    
    bf_DynamicAR() :
      bf_SamplePredictModelReal(NULL){};

    bf_DynamicAR(bf_RandomGenerator *_rng) :
      bf_SamplePredictModelReal(_rng){};

    //-----
    virtual void init()=0;

    //-----
    virtual int nbComponents()=0;

    //-----
    virtual bf_DynamicARModel dynamicModel()=0;

    //-----
    // WRANING : NUMBER of STATES  PER COMPONENT
    // if no number given, return total number of states
    virtual int nbStates(int component=-1)=0;

    //----
    // There is only one noise value per component
    virtual void  setNoiseVariance(int component,real var_noise)=0;


    //---- To Initialize a default state given the values
    //     of the component
    //    state is of length nbStates(), while componentvalues
    //    is of length m_iNbComponents only

    virtual void  initState(real *state,real *componentvalues)=0;

    // 
    //----  INHERITED FUNCTIONS THAT SHOULD BE DEFINED (at least)
    //
    inline virtual void sampleConditionalReal(real *pX, real *pY)=0;
    
    //--------
    inline virtual  void fPredictReal(real *pX, real *pY)=0;

    //--------
    inline virtual  void fNoiseReal(real *pX, real *pY=NULL)=0;

    //-----
    virtual ~bf_DynamicAR(){}
    
  };

  //===================================================
  class bf_DynamicAR2 : virtual public bf_DynamicAR  {
  private:
    bf_RandomGenerator         m_cRng;
    real                      *m_pStdDev;
    int                        m_iNbComponents;

    void create(int nbComponents,real *Var);


  public:
    //-----
    bf_DynamicAR2(int nbComponents,real *Var);

    //-----
    virtual void init();

    //-----
    virtual int nbComponents();

    //-----
    virtual bf_DynamicARModel dynamicModel();

    //-----
    // WARNING : NUMBER of STATES  PER COMPONENT
    // if no number given, return total number of states
    virtual int nbStates(int component=-1);

    //----
    // There is only one noise value per component
    virtual void  setNoiseVariance(int component, real var_noise);

    //---- To Initialize a default state given the values
    //     of the component
    //    state is of length nbStates(), while componentvalues
    //    is of length m_iNbComponents only

    virtual void  initState(real *state,real *componentvalues);

    //--------
    // MEMORY SHOULD BE ALREADY ALLOCATED
    // generating a sample according to P ( X | Y)
    inline virtual void sampleConditionalReal(real *pX, real *pY){
      int i;
      m_cRng.set();
      for(i=0;i<m_iNbComponents;i++){
	*pX=2.* *pY - *(pY+1) + m_pStdDev[i]*m_cRng.gaussian_01(); 
	*(pX+1)=*pY;
	pX+=2; pY+=2;
      }
      m_cRng.release();
    }
    
    //--------
    inline virtual  void fPredictReal(real *pX, real *pY){
      int i;
      for(i=0;i<m_iNbComponents;i++){
	*pX=2.* *pY - *(pY+1); *(pX+1)=*pY;
	pX+=2; pY+=2;
      }
    }

    //--------
    inline virtual  void fNoiseReal(real *pX, real *pY=NULL){
      int i;
      m_cRng.set();
      for(i=0;i<m_iNbComponents;i++){
	*pX=m_pStdDev[i]*m_cRng.gaussian_01();  *(pX+1)=0.;
	pX+=2;
      }
      m_cRng.release();
    }


    //-----
    virtual ~bf_DynamicAR2();


  };

  //===================================================
  class bf_DynamicAR2bis : virtual public bf_DynamicAR  {
  private:
    bf_RandomGenerator         m_cRng;
    real                      *m_pStdDev;
    int                        m_iNbComponents;

    void create(int nbComponents,real *Var);


  public:
    //-----
    bf_DynamicAR2bis(int nbComponents,real *Var);

    //-----
    virtual void init();

    //-----
    virtual int nbComponents();

    //-----
    virtual bf_DynamicARModel dynamicModel();

    //-----
    // WARNING : NUMBER of STATES  PER COMPONENT
    // if no number given, return total number of states
    virtual int nbStates(int component=-1);

    //----
    // There is only one noise value per component
    virtual void  setNoiseVariance(int component, real var_noise);

    //---- To Initialize a default state given the values
    //     of the component
    //    state is of length nbStates(), while componentvalues
    //    is of length m_iNbComponents only

    virtual void  initState(real *state,real *componentvalues);

    //--------
    // MEMORY SHOULD BE ALREADY ALLOCATED
    // generating a sample according to P ( X | Y)
    inline virtual void sampleConditionalReal(real *pX, real *pY){
      int i;
      m_cRng.set();
      for(i=0;i<m_iNbComponents;i++){
	*pX=3.* *pY - 3* *(pY+1) + *(pY+2) + m_pStdDev[i]*m_cRng.gaussian_01(); 
	*(pX+1)=*pY;
	*(pX+2)=*(pY+1);
	pX+=3; pY+=3;
      }
      m_cRng.release();
    }
    
    //--------
    inline virtual  void fPredictReal(real *pX, real *pY){
      int i;
      for(i=0;i<m_iNbComponents;i++){
	*pX=3.* *pY - 3* *(pY+1) + *(pY+2) ;
	*(pX+1)=*pY;
	*(pX+2)=*(pY+1);
	pX+=3; pY+=3;
      }
    }

    //--------
    inline virtual  void fNoiseReal(real *pX, real *pY=NULL){
      int i;
      m_cRng.set();
      for(i=0;i<m_iNbComponents;i++){
	*pX=m_pStdDev[i]*m_cRng.gaussian_01();  *(pX+1)=0.;*(pX+2)=0.;
	pX+=3;
      }
      m_cRng.release();
    }


    //-----
    virtual ~bf_DynamicAR2bis();


  };

  //===================================================
  class bf_DynamicAR3 : virtual public bf_DynamicAR  {
  private:
    bf_RandomGenerator         m_cRng;
    real                      *m_pStdDev;
    real                      *m_pAlpha;
    int                        m_iNbComponents;

    void create(int nbComponents,real *Var,real *alpha=NULL);


  public:
    //-----
    bf_DynamicAR3(int nbComponents,real *Var,real *alpha=NULL);

    //-----
    virtual void init();

    //-----
    virtual int nbComponents();

    //-----
    virtual bf_DynamicARModel dynamicModel();

    //-----
    // WARNING : NUMBER of STATES  PER COMPONENT
    // if no number given, return total number of states
    virtual int nbStates(int component=-1);

    //----
    // There is only one noise value per component
    virtual void  setNoiseVariance(int component, real var_noise);

    //----
    // 
    virtual void  setAlpha(int component, real alpha);

    //---- To Initialize a default state given the values
    //     of the component
    //    state is of length nbStates(), while componentvalues
    //    is of length m_iNbComponents only

    virtual void  initState(real *state,real *componentvalues);

    //--------
    // MEMORY SHOULD BE ALREADY ALLOCATED
    // generating a sample according to P ( X | Y)
    inline virtual void sampleConditionalReal(real *pX, real *pY){
      int i;
      m_cRng.set();
      for(i=0;i<m_iNbComponents;i++){
	*pX=*pY+*(pY+1) + m_pStdDev[i]*m_cRng.gaussian_01();
	//	*pX=2.* *pY - *(pY+1) + m_pStdDev[i]*gaussian_01(); 
	*(pX+1)= (m_pAlpha[i])* *(pY+1)+(1.-m_pAlpha[i])*(*pX-*pY);
	pX+=2; pY+=2;
      }
      m_cRng.release();
    }
    
    //--------
    inline virtual  void fPredictReal(real *pX, real *pY){
      int i;
      for(i=0;i<m_iNbComponents;i++){
	*pX=*pY+*(pY+1);
	//	*pX=2.* *pY - *(pY+1) + m_pStdDev[i]*gaussian_01(); 
	*(pX+1)= (m_pAlpha[i])* *(pY+1)+(1.-m_pAlpha[i])*(*pX-*pY);
	pX+=2; pY+=2;
      }
    }

    //--------
    inline virtual  void fNoiseReal(real *pX, real *pY=NULL){
      int i;
      m_cRng.set();
      for(i=0;i<m_iNbComponents;i++){
	*pX=m_pStdDev[i]*m_cRng.gaussian_01();  *(pX+1)=0.;
	pX+=2;
      }
      m_cRng.release();
    }


    //-----
    virtual ~bf_DynamicAR3();


  };





  //===================================================
  class bf_DynamicAR3bis : virtual public bf_DynamicAR  {
  private:
    bf_RandomGenerator         m_cRng;
    real                      *m_pStdDev;
    real                      *m_pAlpha;
    int                        m_iNbComponents;

    void create(int nbComponents,real *Var,real *alpha=NULL);


  public:
    //-----
    bf_DynamicAR3bis(int nbComponents,real *Var,real *alpha=NULL);

    //-----
    virtual void init();

    //-----
    virtual int nbComponents();

    //-----
    virtual bf_DynamicARModel dynamicModel();

    //-----
    // WARNING : NUMBER of STATES  PER COMPONENT
    // if no number given, return total number of states
    virtual int nbStates(int component=-1);

    //----
    // There is only one noise value per component
    virtual void  setNoiseVariance(int component, real var_noise);

    //----
    // 
    virtual void  setAlpha(int component, real alpha);

    //---- To Initialize a default state given the values
    //     of the component
    //    state is of length nbStates(), while componentvalues
    //    is of length m_iNbComponents only

    virtual void  initState(real *state,real *componentvalues);

    //--------
    // MEMORY SHOULD BE ALREADY ALLOCATED
    // generating a sample according to P ( X | Y)
    inline virtual void sampleConditionalReal(real *pX, real *pY){
      int i;
      m_cRng.set();
      for(i=0;i<m_iNbComponents;i++){
	*pX=*pY+*(pY+1) + *(pY+2)*0.5 + m_pStdDev[i]*m_cRng.gaussian_01();
	*(pX+1)= *(pY+1) + *(pY+2);
	*(pX+2)= (m_pAlpha[i])* *(pY+2)+(1.-m_pAlpha[i])*((*pX-*pY)-*(pY+1));
	pX+=3; pY+=3;
      }
      m_cRng.release();
    }
    
    //--------
    inline virtual  void fPredictReal(real *pX, real *pY){
      int i;
      for(i=0;i<m_iNbComponents;i++){
	*pX=*pY+*(pY+1) + *(pY+2)*0.5;
	*(pX+1)= *(pY+1) + *(pY+2);
	*(pX+2)= (m_pAlpha[i])* *(pY+2)+(1.-m_pAlpha[i])*((*pX-*pY)-*(pY+1));
	pX+=3; pY+=3;
      }
    }

    //--------
    inline virtual  void fNoiseReal(real *pX, real *pY=NULL){
      int i;
      m_cRng.set();
      for(i=0;i<m_iNbComponents;i++){
	*pX=m_pStdDev[i]*m_cRng.gaussian_01();  *(pX+1)=0.; *(pX+2)=0.;
	pX+=2;
      }
      m_cRng.release();
    }


    //-----
    virtual ~bf_DynamicAR3bis();


  };


}

#endif
