// this file does not really use open cv
// however, to keep consistent notation, i have labeled
// this class with ipcv_ prefix

#include "ipcv_ContourTemplate.h"


namespace ImageProcessing {


  //-----
  ipcv_ContourTemplate::ipcv_ContourTemplate()
  {
    m_iNumberOfPoints = 0;
    m_pPointList = NULL;
    m_pValidPoint = NULL;
    m_pNormalVectorList = NULL;
    m_pNormalLineList = NULL;
    m_iMeasureType = MEASURE_BLAKE;
    m_pShape = NULL;
    m_rLineLength = 0;
    m_iNumOfPointsInLine = 0;
  }


  //-----
  ipcv_ContourTemplate::ipcv_ContourTemplate(int numOfPoints, 
					     int measureType,
					     real lineLength,
					     int numberOfPointsOnLine)
  {
    create(numOfPoints, measureType, lineLength, numberOfPointsOnLine);
  }


  //-----
  void ipcv_ContourTemplate::create(int numOfPoints, 
				    int measureType,
				    real lineLength,
				    int numberOfPointsOnLine)
  { 
    m_iNumberOfPoints = numOfPoints;
    m_pPointList = new ip_Point[m_iNumberOfPoints];   
    m_pValidPoint = new bool[m_iNumberOfPoints];
    for(int i=0;i<m_iNumberOfPoints;i++)
      m_pValidPoint[i] = false;
    m_pNormalVectorList = new ip_Point[m_iNumberOfPoints];
    m_pNormalLineList = new ip_LineSegment[m_iNumberOfPoints];
    m_iMeasureType = measureType;
    m_pShape = NULL;
    m_rLineLength = lineLength;
    m_iNumOfPointsInLine = numberOfPointsOnLine;
  }

  //----- assuming that points on contour have been already extracted
  void ipcv_ContourTemplate::validatePointsOnContour(int rows, int cols)
  {
    for(int i=0;i<m_iNumberOfPoints;i++)
      m_pValidPoint[i] = ip_inImage(m_pPointList[i], rows, cols);
  }

  //----- 
  void ipcv_ContourTemplate::computeLineSegments(ip_Shape *pShape, int rows, int cols)
  {
    m_pShape = pShape;
    real stepSize = 6.28318 / m_iNumberOfPoints;
    for(int i=0;i<m_iNumberOfPoints;i++)
      if(m_pValidPoint[i] == true)
      {     	
	m_pShape->computeNormalToPoint(m_pNormalVectorList+i,stepSize*i);	
	m_pNormalLineList[i].init(m_pPointList+i,m_pNormalVectorList+i,m_rLineLength, m_iNumOfPointsInLine, rows, cols);
      }
  }

  
  //-----
  void ipcv_ContourTemplate::computeEdgeFeatures(ip_Image<ip_ColorElement8u> *pImage,
						 int sigma, real threshold)
  { 
    for(int i=0;i<m_iNumberOfPoints;i++)
      if(m_pValidPoint[i] == true)
      {
	//create a vector for processing 
	ip_1D_Vector pSignalIn;
	pSignalIn.create(m_pNormalLineList[i].m_iNumberOfPoints);
	//printf("i:%hd\n",i);
	ip_ComputeEdgeFeatures(m_pNormalLineList+i,pImage,&pSignalIn,sigma,threshold);
	pSignalIn.freeMemory();
      }   
  }  
 



  //----- all process for edge feature extraction in one step!
  void ipcv_ContourTemplate::extractEdgeFeatures(ip_Shape *pShape, int rows, int cols, 
						 ip_Image<ip_ColorElement8u> *pImage,
						 int sigma, real threshold)
  { 
    m_pShape = pShape;
    real stepSize = 6.28318 / m_iNumberOfPoints;
    for(int i=0;i<m_iNumberOfPoints;i++)
    {    
      //validatePointsOnContour
      m_pValidPoint[i] = ip_inImage(m_pPointList[i], rows, cols);
      if(m_pValidPoint[i] == true)
      {     	
	//computeLineSegments
	m_pShape->computeNormalToPoint(m_pNormalVectorList+i,stepSize*i);	
	m_pNormalLineList[i].init(m_pPointList+i,m_pNormalVectorList+i,m_rLineLength, m_iNumOfPointsInLine, rows, cols);
	
	//computeEdgeFeatures
	//create a vector for processing 
	ip_1D_Vector pSignalIn;
	pSignalIn.create(m_pNormalLineList[i].m_iNumberOfPoints);
	//printf("i:%hd\n",i);
	ip_ComputeEdgeFeatures(m_pNormalLineList+i,pImage,&pSignalIn,sigma,threshold);
	pSignalIn.freeMemory();
      }  
    } 
  }  
 




  
  /* problem with this likelihood: it does not provide enough discrimination
     with respect to how good the weights really are...
  //original version by isard: see his thesis, pp48
  //----    
  real ipcv_ContourTemplate::computeLikelihood(bf_EvalDistReal *pProb)
  {
    real globalLikelihood = 1.;
    real localLikelihood, d;

    for(int i=0;i<m_iNumberOfPoints;i++)
      if(m_pValidPoint[i] == true)
      {
	localLikelihood = 1.;
	for(int j=0;j<m_pNormalLineList[i].m_iNumberOfFeatures;j++)
	{
	  d = ip_euclideanDistance(m_pPointList[i], m_pNormalLineList[i].m_pFeatures[j]);
	  //printf("d = %f\n",d);
	  localLikelihood += pProb->evaluateReal(&d);	
	}
	globalLikelihood *= localLikelihood;
      }  
    //printf("Likelihood:%f\n",globalLikelihood);
    return globalLikelihood;
  }
  */
 

  //----  
  // simplified version: only one point per line: (isard's thesis, pp. 50. the minimum distance)
  // the problem with this likelihood is that when a line does not find any point,
  // the likelihood value decreases considerably: not very robust to lack of detected points
  // additionally, the numbers gets too small too quickly, but the use of logs 
  // again affects the way in which weights are taken into account in the particle filter ...
  // MODIFICATION: to deal with image boundaries, we do not know whether the contour would
  // "hit" any edges if it was extrapolated. if we penalize and give the smaller value, contours
  // will likely get lost. on the other hand, if we simply ignore those terms (assuming that the
  // local likelihood is one, we will weight such configurations much more.
  // the solution is to randomly assign them a "hitting" value in the local likelihood.

  /*
  //----
  real ipcv_ContourTemplate::computeLikelihood(bf_EvalDistReal *pProb)
  {
    real globalLikelihood = 1.;
    real localLikelihood, d, minima;

    for(int i=0;i<m_iNumberOfPoints;i++)
    {
      if(m_pValidPoint[i] == true)
      {	
	minima = MIN_DIST_DEF;
	for(int j=0;j<m_pNormalLineList[i].m_iNumberOfFeatures;j++)
	{
	  d = ip_euclideanDistance(m_pPointList[i], m_pNormalLineList[i].m_pFeatures[j]);
	  if(d < minima)
	    minima = d;
	}
	localLikelihood = pProb->evaluateReal(&minima);		
	globalLikelihood *= localLikelihood;
      }
      else
      {
	minima = max(MIN_DIST_DEF * uniform(), MIN_DIST_DEF/2);
	localLikelihood = pProb->evaluateReal(&minima);		
	globalLikelihood *= localLikelihood;
      }
    }
    //printf("Likelihood:%f\n",globalLikelihood);
    return globalLikelihood;
  } 
  */


  //----
  // option 2: normalize by the number of points
  real ipcv_ContourTemplate::computeLikelihood(bf_EvalDistReal *pProb)
  {
    real globalLikelihood = 1.;
    real localLikelihood, d, minima;

    int iValidPoints = 0;
    for(int i=0;i<m_iNumberOfPoints;i++)
		{
		  if(m_pValidPoint[i] == true)
			 {	
				minima = MIN_DIST_DEF;
				for(int j=0;j<m_pNormalLineList[i].m_iNumberOfFeatures;j++)
				  {
					 d = ip_euclideanDistance(m_pPointList[i], m_pNormalLineList[i].m_pFeatures[j]);
					 if(d < minima)
						minima = d;
				  }
				localLikelihood = pProb->evaluateReal(&minima);		
				globalLikelihood *= localLikelihood;
				iValidPoints++;
			 }
		}
	 //    if(iValidPoints!=0)
    //  return globalLikelihood/float(iValidPoints);
    if(iValidPoints!=0)
      return (real)pow((double)globalLikelihood,8./double(iValidPoints));
    else
      return 0.;
  } 



  //----- 
  void ipcv_ContourTemplate::drawLineSegments(ip_Image<ip_ColorElement8u> & Ima,
					      ip_ColorElement8u color=ip_GREEN)
  {
    for(int i=0;i<m_iNumberOfPoints;i++)
      if(m_pValidPoint[i] == true)
	ip_bresenham( &(m_pNormalLineList[i].m_StartPoint),
		      &(m_pNormalLineList[i].m_EndPoint), Ima, color);
   
  } 


  //----- 
  void ipcv_ContourTemplate::drawLineSegments(ip_Image<ip_ColorElement8u> *pImage,
					      ip_ColorElement8u color=ip_GREEN)
  {
    for(int i=0;i<m_iNumberOfPoints;i++)
      if(m_pValidPoint[i] == true)
      {
	//	ip_bresenham( &(m_pNormalLineList[i].m_StartPoint),
	//	      &(m_pNormalLineList[i].m_EndPoint), pImage, color);
	ip_safeDraw( &(m_pNormalLineList[i].m_StartPoint),
		     &(m_pNormalLineList[i].m_EndPoint), pImage, color);
      }   

  } 


  //-----
  void ipcv_ContourTemplate::drawFeatures(ip_Image<ip_ColorElement8u> *pImage, 
					  ip_ColorElement8u color=ip_MAGENTA)
  { 
    for(int i=0;i<m_iNumberOfPoints;i++)
      if(m_pValidPoint[i] == true) 
	ip_DrawDetectedPoints(m_pNormalLineList+i, pImage, color);      
  }  


  //-----
  void ipcv_ContourTemplate::freeMemory(void)
  {
    
    if(m_pPointList != NULL)
    {
      delete[] m_pPointList;
      m_pPointList = NULL;
    }
    if(m_pValidPoint != NULL)
    {
      delete[] m_pValidPoint;
      m_pValidPoint = NULL;
    }
    if(m_pNormalVectorList != NULL)
    {
      delete[] m_pNormalVectorList;
      m_pNormalVectorList = NULL;
    }
    if(m_pNormalLineList != NULL)
    {
      delete[] m_pNormalLineList;
      m_pNormalLineList = NULL;
    }        
  }


}

