#include "general.h"
#include "ip_ImageFeatureFunctions.h"


namespace ImageProcessing {


/**************************************************************************
ip_Round() 
version: 05.25.02 
**************************************************************************/
int ip_Round(float x)
{ 
  if(x-(int)x >0.5)
    return (int)x + 1;
  else 
    return (int)x;
}


/**************************************************************************
ip_rgb2y.cc 
version: 05.25.02 
**************************************************************************/
real ip_rgb2y(int x, int y, ip_Image<ip_ColorElement8u> &Image)
{
  Image.setLine(y);
  return 0.299*Image.valueCol(x).one 
       + 0.587*Image.valueCol(x).two
       + 0.114*Image.valueCol(x).three;

} /* ip_rgb2y() */


/**************************************************************************
ip_rgb2y.cc Overloaded function that uses pointer to
an image instead of a reference.

version: 05.30.02 
**************************************************************************/
real ip_rgb2y(int x, int y, ip_Image<ip_ColorElement8u> *pImage)
{
  pImage->setLine(y);
  return 0.299*pImage->valueCol(x).one 
       + 0.587*pImage->valueCol(x).two
       + 0.114*pImage->valueCol(x).three;

} /* ip_rgb2y() */


/**************************************************************************
ip_dataAt.cc 

version: 05.30.02 
**************************************************************************/
real ip_dataAt(int x, int y, int band, ip_Image<ip_ColorElement8u> *pImage)
{
  pImage->setLine(y);
  if(band == 1)
    return pImage->valueCol(x).one;
  else if(band == 2)
    return pImage->valueCol(x).two;
  else
    return pImage->valueCol(x).three;
      
} /* ip_dataAt() */



/**************************************************************************
ip_bilinearInterpolation.cc. Assumes processing on the Y band of 
a color image, so each pixel that is used in the process is converted 
on the fly.
version: 05.25.02 
**************************************************************************/
real ip_bilinearInterpolation(ip_Point *pPoint, 
			      ip_Image<ip_ColorElement8u> &Image)
{
  real x_frac = pPoint->m_rX - (int)pPoint->m_rX; 
  real y_frac = pPoint->m_rY - (int)pPoint->m_rY; 

  real a = (1-x_frac)*(1-y_frac);
  real b = (1-x_frac)*y_frac;
  real c = x_frac*(1-y_frac);
  real d = x_frac*y_frac;


  return a*ip_rgb2y((int)pPoint->m_rX, (int)pPoint->m_rY, Image)
    + b*ip_rgb2y((int)pPoint->m_rX, (int)pPoint->m_rY+1, Image)
    + c*ip_rgb2y((int)pPoint->m_rX+1, (int)pPoint->m_rY, Image)
    + d*ip_rgb2y((int)pPoint->m_rX+1, (int)pPoint->m_rY+1, Image);


} /* ip_bilinearInterpolation() */


/**************************************************************************
ip_bilinearInterpolation.cc. Overloaded function that uses pointer to
an image instead of a reference.

version: 05.30.02 
**************************************************************************/
real ip_bilinearInterpolation(ip_Point *pPoint, 
			      ip_Image<ip_ColorElement8u> *pImage)
{

  int  x = (int)pPoint->m_rX;
  int  y = (int)pPoint->m_rY;

  real x_frac = pPoint->m_rX - x;
  real y_frac = pPoint->m_rY - y;
  

  real a = (1-x_frac)*(1-y_frac);
  real b = (1-x_frac)*y_frac;
  real c = x_frac*(1-y_frac);
  real d = x_frac*y_frac;


#define  SAFE_BILINEAR
#ifdef SAFE_BILINEAR  
  if(x>=0 && y>=0 && x+1<pImage->nbColumns() && y+1 < pImage->nbLines())
#endif	 
	 {
		if(OPTION_PREPROCESSING == ON_LINE_RGB2Y)
		  {
			 /*		  return a*ip_rgb2y((int)pPoint->m_rX, (int)pPoint->m_rY, pImage)
						  + b*ip_rgb2y((int)pPoint->m_rX, (int)pPoint->m_rY+1, pImage)
						  + c*ip_rgb2y((int)pPoint->m_rX+1, (int)pPoint->m_rY, pImage)
						  + d*ip_rgb2y((int)pPoint->m_rX+1, (int)pPoint->m_rY+1, pImage);*/
			 return a*ip_rgb2y(x,y, pImage)	 + b*ip_rgb2y(x,y+1, pImage)
				+ c*ip_rgb2y(x+1,y, pImage)	 + d*ip_rgb2y(x+1,y+1, pImage);
		  }
		else
		  {  
			 return a*ip_dataAt((int)pPoint->m_rX, (int)pPoint->m_rY, 1, pImage)
				+ b*ip_dataAt((int)pPoint->m_rX, (int)pPoint->m_rY+1, 1, pImage)
				+ c*ip_dataAt((int)pPoint->m_rX+1, (int)pPoint->m_rY, 1, pImage)
				+ d*ip_dataAt((int)pPoint->m_rX+1, (int)pPoint->m_rY+1, 1, pImage);
			 
		  }
	 }
#ifdef SAFE_BILINEAR  
  else{
	 printf("WARNING BASE POINT IN INTERPOLATION OUT OF IMAGE : li=%d co =%d\n",y,x);
	 fflush(stdout);
	 return 0.;
	 
  }
#endif	 
  

} /* ip_bilinearInterpolation() */


/**************************************************************************
ip_bilinearInterpolation.cc.Directly uses a Y image.

version: 06.21.02 
**************************************************************************/
real ip_bilinearInterpolation(ip_Point *pPoint, unsigned char *pImage, int cols)
{

  real x_frac = pPoint->m_rX - (int)pPoint->m_rX; 
  real y_frac = pPoint->m_rY - (int)pPoint->m_rY; 

  real a = (1-x_frac)*(1-y_frac);
  real b = (1-x_frac)*y_frac;
  real c = x_frac*(1-y_frac);
  real d = x_frac*y_frac;

  return a * pImage[(int)pPoint->m_rX + (int)pPoint->m_rY*cols]
    + b * pImage[(int)pPoint->m_rX + ((int)pPoint->m_rY+1)*cols]
    + c * pImage[(int)pPoint->m_rX+1 + (int)pPoint->m_rY*cols]
    + d * pImage[(int)pPoint->m_rX+1 + ((int)pPoint->m_rY+1)*cols];

} /* ip_bilinearInterpolation() */



/**************************************************************************
ip_generateLineOfMeasurements.cc. Computes the interpolated intensity values
for all the desired points on a line segment.The line is from outside to 
inside, so the normal vector has to be multiplied by -1.
version: 05.25.02 
**************************************************************************/
void ip_generateLineOfMeasurements(ip_LineSegment *pLine,
				   ip_Image<ip_ColorElement8u> &Image,
				   ip_1D_Vector *pData)
{

  //measurement points are evenly spread along the line
  float stepSize = pLine->m_dLength / pLine->m_iNumberOfPoints;

  ip_Point point;
  for(int i=0;i<pLine->m_iNumberOfPoints;i++)
  {    
    point.m_rX = pLine->m_StartPoint.m_rX - pLine->m_Normal.m_rX*i*stepSize;
    point.m_rY = pLine->m_StartPoint.m_rY - pLine->m_Normal.m_rY*i*stepSize;
    pData->put(i,ip_bilinearInterpolation(&point,Image));
  }

} /* ip_generateLineOfMeasurements() */


/**************************************************************************
ip_generateLineOfMeasurements.cc. Overloaded function that uses pointer to
an image instead of a reference.
version: 05.30.02 
**************************************************************************/
void ip_generateLineOfMeasurements(ip_LineSegment *pLine,
				   ip_Image<ip_ColorElement8u> *pImage,
				   ip_1D_Vector *pData)
{

  //measurement points are evenly spread along the line
  float stepSize = pLine->m_dLength / pLine->m_iNumberOfPoints;

  ip_Point point;
  for(int i=0;i<pLine->m_iNumberOfPoints;i++)
  {    
    point.m_rX = pLine->m_StartPoint.m_rX - pLine->m_Normal.m_rX*i*stepSize;
    point.m_rY = pLine->m_StartPoint.m_rY - pLine->m_Normal.m_rY*i*stepSize;
    pData->put(i,ip_bilinearInterpolation(&point, pImage));
  }

} /* ip_generateLineOfMeasurements() */



/**************************************************************************
ip_generateLineOfMeasurements.cc. Function that uses an already Y image
version: 06.21.02 
**************************************************************************/
void ip_generateLineOfMeasurements(ip_LineSegment *pLine,
				   unsigned char *pImage, int cols,
				   ip_1D_Vector *pData)
{

  //measurement points are evenly spread along the line
  float stepSize = pLine->m_dLength / pLine->m_iNumberOfPoints;

  ip_Point point;
  for(int i=0;i<pLine->m_iNumberOfPoints;i++)
  {    
    point.m_rX = pLine->m_StartPoint.m_rX - pLine->m_Normal.m_rX*i*stepSize;
    point.m_rY = pLine->m_StartPoint.m_rY - pLine->m_Normal.m_rY*i*stepSize;
    pData->put(i,ip_bilinearInterpolation(&point, pImage,cols));
  }

} /* ip_generateLineOfMeasurements() */






/**************************************************************************
ip_Create1DGaussianFilter. sigma parameter is assumed integer.
memory is created here...
h(i)=exp(-i.^2/2*sigma.^2)
version: 05.25.02 
**************************************************************************/
void ip_Create1DGaussianFilter(ip_1D_Vector *pFilter, int sigma)
{
  int i;
  real data,norm;

  pFilter->create(2*2*sigma+1); 
  norm=0.;
  for(i=-2*sigma; i<=2*sigma; i++) 
  {
    data =  exp(-(i*i)/(2*sigma*sigma));
	 norm+=data;
    // the center of the kernel is in 2*sigma 
    pFilter->put(i+2*sigma, data);
  }
  for(i=0; i<4*sigma+1; i++) 
  {
    pFilter->put(i,pFilter->get(i)/norm);
  }
} /* ip_Create1DGaussianFilter() */


/**************************************************************************
ip_Convolve1DGaussianFilter() 
version: 05.25.02 
**************************************************************************/
void ip_Convolve1DGaussianFilter(ip_1D_Vector *pSignalIn, 
				 ip_1D_Vector *pFilter,
				 ip_1D_Vector *pSignalOut)
{ 
  int size = pFilter->m_iSize/2;
  real add;

  for(int index=0; index<pSignalIn->m_iSize; index++)
  {
    add=0.0;
    for(int i=-size; i<=size; i++)     
      add = add + pFilter->get(i+size) * pSignalIn->get(index+i);
    pSignalOut->put(index,add);
  }

} /* ip_Convolve1DGaussianFilter() */
      
    
/**************************************************************************
ip_1DGradient() 
version: 05.25.02 
**************************************************************************/
void ip_1DGradient(ip_1D_Vector *pSignalIn, ip_1D_Vector *pSignalOut)
{ 
  for(int i=0; i<pSignalIn->m_iSize; i++)
    pSignalOut->put(i, fabs((pSignalIn->get(i+1) - pSignalIn->get(i-1))/2));
    
} /* ip_1DGradient() */
     

/**************************************************************************
ip_1DThreshold() 
version: 05.25.02 
**************************************************************************/
void ip_1DThreshold(ip_1D_Vector *pSignalIn, 
		    ip_1D_Vector *pSignalOut,
		    float threshold)
{ 
  for(int i=0; i<pSignalIn->m_iSize; i++)
  {
    if( pSignalIn->get(i) > threshold )
      pSignalOut->put(i, IP_EDGE_PIXEL);
    else
      pSignalOut->put(i, IP_NON_EDGE_PIXEL);
  }

} /* ip_1DThreshold() */
     

/**************************************************************************
ip_1DNonMaximumSupression() thin edges... 
version: 05.25.02 
**************************************************************************/
void ip_1DNonMaximumSupression(ip_1D_Vector *pSignalGrad, 
			       ip_1D_Vector *pSignalEdges, 
			       ip_1D_Vector *pSignalOut)
{ 
  for(int i=0; i<pSignalEdges->m_iSize-1; i++)
	 {
		if( pSignalEdges->get(i) == IP_EDGE_PIXEL)
		  {
			 if( pSignalGrad->get(i) > pSignalGrad->get(i+1) &&
				  pSignalGrad->get(i) > pSignalGrad->get(i-1) )
				pSignalOut->put(i, IP_EDGE_PIXEL);
			 else
				pSignalOut->put(i, IP_NON_EDGE_PIXEL);
		  }
		else
		  pSignalOut->put(i, IP_NON_EDGE_PIXEL);
	 }
  
} /* ip_1DNonMaximumSupression() */
     

/**************************************************************************
ip_1DNonMaximumSupression() the ultimate version 
version: 06.02.02 
**************************************************************************/
void ip_1DNonMaximumSupression(ip_1D_Vector *pSignalGrad, 
			       ip_1D_Vector *pSignalEdges, 
			       ip_LineSegment *pLine)
{ 

  //measurement points are evenly spread along the line
  float stepSize = pLine->m_dLength / pLine->m_iNumberOfPoints;

  for(int i=0; i<pSignalEdges->m_iSize-1; i++)
	 {
		if( pSignalEdges->get(i) == IP_EDGE_PIXEL)
		  {
			 if( pSignalGrad->get(i) > pSignalGrad->get(i+1) &&
				  pSignalGrad->get(i) > pSignalGrad->get(i-1) )
				{
				  pLine->m_pFeatures[pLine->m_iNumberOfFeatures].m_rX = 
					 (pLine->m_StartPoint.m_rX-pLine->m_Normal.m_rX*i*stepSize);
				  pLine->m_pFeatures[pLine->m_iNumberOfFeatures].m_rY = 
					 (pLine->m_StartPoint.m_rY-pLine->m_Normal.m_rY*i*stepSize);
				  pLine->m_iNumberOfFeatures++;	
				}
		  }
	 }

} /* ip_1DNonMaximumSupression() */
   

/**************************************************************************
ip_1DEdgeDetection() Returns a vector with detected features.
version: 05.25.02 
**************************************************************************/
void ip_1DEdgeDetection(ip_1D_Vector *pSignalIn, 
								ip_1D_Vector *pSignalOut, 
								int sigma, 
								real threshold)
{ 
  ip_1D_Vector filter, temp, temp2;
  temp.create(pSignalIn->m_iSize);
  temp2.create(pSignalIn->m_iSize);
  
  ip_Create1DGaussianFilter(&filter, sigma);
  ip_Convolve1DGaussianFilter(pSignalIn, &filter, &temp);
  ip_1DGradient(&temp, &temp2);
  ip_1DThreshold(&temp2, &temp, threshold);
  ip_1DNonMaximumSupression(&temp2, &temp, pSignalOut);
 
  temp.freeMemory();
  temp2.freeMemory();
  filter.freeMemory();

} /* ip_1DEdgeDetection() */


/**************************************************************************
ip_1DEdgeDetection() the ultimate version that copies directly to the
line segment object 
version: 06.02.02 
**************************************************************************/
void ip_1DEdgeDetection(ip_1D_Vector *pSignalIn, 
			ip_LineSegment *pLine, 
			int sigma, 
			real threshold)
{ 
  ip_1D_Vector filter, temp, temp2;
  temp.create(pSignalIn->m_iSize);
  temp2.create(pSignalIn->m_iSize);

  ip_Create1DGaussianFilter(&filter, sigma);
  ip_Convolve1DGaussianFilter(pSignalIn, &filter, &temp);
  ip_1DGradient(&temp, &temp2);
  ip_1DThreshold(&temp2, &temp, threshold);
  ip_1DNonMaximumSupression(&temp2, &temp, pLine);
 
  temp.freeMemory();
  temp2.freeMemory();
  filter.freeMemory();

} /* ip_1DEdgeDetection() */
   

/**************************************************************************
ip_ComputeEdgeFeatures(): given a line segment, it finds the features
version: 05.25.02 
**************************************************************************/
void ip_ComputeEdgeFeatures(ip_LineSegment *pLine, 
			    ip_Image<ip_ColorElement8u> &Image, 
			    ip_1D_Vector *pSignalIn,
			    ip_1D_Vector *pSignalOut,
			    int sigma, real threshold)
{ 
  ip_generateLineOfMeasurements(pLine, Image, pSignalIn);
  ip_1DEdgeDetection(pSignalIn, pSignalOut, sigma, threshold);

} /* ip_ComputeEdgeFeatures() */

   
/**************************************************************************
ip_ComputeEdgeFeatures(): Overloaded function that uses pointer to
an image instead of a reference.
**************************************************************************/
void ip_ComputeEdgeFeatures(ip_LineSegment *pLine, 
			    ip_Image<ip_ColorElement8u> *pImage, 
			    ip_1D_Vector *pSignalIn,
			    ip_1D_Vector *pSignalOut,
			    int sigma, real threshold)
{ 
  ip_generateLineOfMeasurements(pLine, pImage, pSignalIn);
  ip_1DEdgeDetection(pSignalIn, pSignalOut, sigma, threshold);

} /* ip_ComputeEdgeFeatures() */
  

/**************************************************************************
ip_ComputeEdgeFeatures(): second version. it stores the detected features in 
pLine as necessary
version: 06.02.02
**************************************************************************/
void ip_ComputeEdgeFeatures(ip_LineSegment *pLine, 
			    ip_Image<ip_ColorElement8u> *pImage, 
			    ip_1D_Vector *pSignalIn,
			    int sigma, real threshold)
{ 
  ip_generateLineOfMeasurements(pLine, pImage, pSignalIn);
  ip_1DEdgeDetection(pSignalIn, pLine, sigma, threshold);
  
} /* ip_ComputeEdgeFeatures() */
  


/**************************************************************************
ip_ComputeEdgeFeatures(): last version, receives an already converted Y image
(stored as unsigned char) 
version: 06.21.02
**************************************************************************/
void ip_ComputeEdgeFeatures(ip_LineSegment *pLine, 
			    unsigned char *pImage, int cols,
			    ip_1D_Vector *pSignalIn,
			    int sigma, real threshold)
{ 
  ip_generateLineOfMeasurements(pLine, pImage, cols, pSignalIn);
  ip_1DEdgeDetection(pSignalIn, pLine, sigma, threshold);
  
} /* ip_ComputeEdgeFeatures() */
 





/**************************************************************************
ip_DrawDetectedPoints() 
version: 05.25.02 
**************************************************************************/
void ip_DrawDetectedPoints(ip_LineSegment *pLine, 
			   ip_1D_Vector *pSignal,
			   ip_Image<ip_ColorElement8u> &Image, 
			   ip_ColorElement8u color=ip_GREEN)
{ 
  //measurement points are evenly spread along the line
  float stepSize = pLine->m_dLength / pLine->m_iNumberOfPoints;
  int x,y;

  for(int i=0;i<pLine->m_iNumberOfPoints;i++)
  {  
    if( pSignal->get(i) == IP_EDGE_PIXEL)    
    {
      float xf=(pLine->m_StartPoint.m_rX-pLine->m_Normal.m_rX*i*stepSize);
      float yf=(pLine->m_StartPoint.m_rY-pLine->m_Normal.m_rY*i*stepSize);
      x = ip_Round(xf); 
      y = ip_Round(yf);
      // draw a 3x3 block around (x,y) in the desired color
      for(int j=-1;j<=1;j++)
	for(int k=-1;k<=1;k++)
	  Image(y+j,x+k) = color;
    }
  }

} /* ip_DrawDetectedPoints() */


/**************************************************************************
ip_DrawDetectedPoints() Overloaded function that uses pointer to
an image instead of a reference.
version: 05.25.02 
**************************************************************************/
void ip_DrawDetectedPoints(ip_LineSegment *pLine, 
			   ip_1D_Vector *pSignal,
			   ip_Image<ip_ColorElement8u> *pImage, 
			   ip_ColorElement8u color=ip_GREEN)
{ 
  //measurement points are evenly spread along the line
  float stepSize = pLine->m_dLength / pLine->m_iNumberOfPoints;
  int x,y;

  for(int i=0;i<pLine->m_iNumberOfPoints;i++)
  {  
    if( pSignal->get(i) == IP_EDGE_PIXEL)    
    {
      float xf=(pLine->m_StartPoint.m_rX-pLine->m_Normal.m_rX*i*stepSize);
      float yf=(pLine->m_StartPoint.m_rY-pLine->m_Normal.m_rY*i*stepSize);
      x = ip_Round(xf); 
      y = ip_Round(yf);
      // draw a 3x3 block around (x,y) in the desired color
      for(int j=-1;j<=1;j++)
	for(int k=-1;k<=1;k++)	  
	  pImage->setVal(y+j, x+k, color);     
    }
  }

} /* ip_DrawDetectedPoints() */



/**************************************************************************
ip_DrawDetectedPoints() version that reads directly from LineSegment
version: 06.02.02 
**************************************************************************/
void ip_DrawDetectedPoints(ip_LineSegment *pLine, 			   
			   ip_Image<ip_ColorElement8u> *pImage, 
			   ip_ColorElement8u color=ip_GREEN)
{ 
  for(int i=0;i<pLine->m_iNumberOfFeatures;i++)
  {  
      int x = ip_Round(pLine->m_pFeatures[i].m_rX); 
      int y = ip_Round(pLine->m_pFeatures[i].m_rY);
      // draw a 3x3 block around (x,y) in the desired color
      for(int j=-1;j<=1;j++)
	for(int k=-1;k<=1;k++)	
	  if( (x+k)>= 0 && (x+k)<pImage->nbColumns() 
	      && (y+j)>= 0 && (y+j)<pImage->nbLines() )
	    pImage->setVal(y+j, x+k, color);     
  }

} /* ip_DrawDetectedPoints() */

} //namespace Torch












