#include "general.h"
#include <stdlib.h>
#include <stdio.h>
#include <iostream.h>
#include <iomanip.h>

#include "ip_Image.h"
#include "ip_ColorDef.h"
#include "ip_PixelAndPoint.h"
#include "ip_DrawFunctions.h"




namespace ImageProcessing {

/**************************************************************************
ip_sign.cc 
version: 05.16.02 
**************************************************************************/
int ip_sign(int datum)
{
   if(datum>0)
      return(1);
   else if(datum<0)
      return(-1);
   else return(0);

} /* sign() */


/**************************************************************************
ip_euclideanDistance.cc 
version: 05.16.02 
**************************************************************************/
real ip_euclideanDistance(ip_Point p1, ip_Point p2)
{

  return sqrt( (p1.m_rX-p2.m_rX)*(p1.m_rX-p2.m_rX) + 
	       (p1.m_rY-p2.m_rY)*(p1.m_rY-p2.m_rY) );

} /* ip_euclideanDistance() */


/**************************************************************************
ip_unitaryDifferenceVector(). Given points p1 and p2, it returns the unitary
vector in the direction of p2-p1
version: 05.16.02 
**************************************************************************/
void ip_unitaryDifferenceVector(ip_Point *p1, ip_Point *p2, ip_Point *pOut)
{
    real mag = ip_euclideanDistance(*p1, *p2);
    if (mag!=0)
    {
      pOut->m_rX = (p2->m_rX - p1->m_rX)/mag;
      pOut->m_rY = (p2->m_rY - p1->m_rY)/mag;
    }
    else
    { 
      pOut->m_rX = 0.;
      pOut->m_rY = 0.;
    }

} /* ip_unitaryDifferenceVector() */



/**************************************************************************
ip_inImage.cc 
version: 05.16.02 
**************************************************************************/
bool ip_inImage(ip_Point p, int rows, int cols)
{
  //  if( p.m_rX>=0 && p.m_rX<cols && p.m_rY>=0 && p.m_rY<rows)
  //  for reals : valid point is in when ? (note max int line : rows-1)
  if( p.m_rX>=0 && p.m_rX<=cols-1 && p.m_rY>=0 && p.m_rY<=rows-1)
    return true;
  else
    return false;
}

/**************************************************************************
ip_inImage.cc 
version: 05.16.02 
**************************************************************************/
bool ip_inImage(ip_Point *p, int rows, int cols)
{
  //  if( p->m_rX>=0 && p->m_rX<cols && p->m_rY>=0 && p->m_rY<rows)
  if( p->m_rX>=0 && p->m_rX<cols-1 && p->m_rY>=0 && p->m_rY<rows-1)
    return true;
  else
    return false;
} 



/**************************************************************************
ip_angle.cc computes tangent of a directed line segment
version: 05.16.02 
**************************************************************************/
real ip_angle(ip_Point p1, ip_Point p2)
{
  real num = p1.m_rY - p2.m_rY;
  real den = p1.m_rX - p2.m_rX;

  if(den != 0)
    return (real) atan2(num,den);
  else if(num > 0)
    return 3.141592/2.;
  else if(num < 0)
    return -3.141592/2.;
  
} /* ip_angle() */


/**************************************************************************
ip_ExtremePoint.cc. Notice the order between the points. Assumes P1 is in
and P2 might be out
version: 06.06.02
**************************************************************************/
void ip_extremePoint(ip_Point *pP1,ip_Point *pP2,ip_Point *pPOut,
		     int rows,int cols)
{
  bool a = ip_inImage(*pP1, rows, cols);
  bool b = ip_inImage(*pP2, rows, cols);
  int i;

  ip_Point vector;

  if( a && b )
    *pPOut = *pP2;
  else if (a && !b)
  {
    ip_unitaryDifferenceVector(pP1, pP2, &vector);
    pPOut->m_rX = pP1->m_rX;
    pPOut->m_rY = pP1->m_rY;
    i=0;
    while(ip_inImage(*pPOut,rows, cols))
    {
      pPOut->m_rX =  pP1->m_rX + i*vector.m_rX;
      pPOut->m_rY =  pP1->m_rY + i*vector.m_rY;
      i++;
    }
    i--;
    pPOut->m_rX =  pP1->m_rX + i*vector.m_rX;
    pPOut->m_rY =  pP1->m_rY + i*vector.m_rY;
  }
}


/**************************************************************************
ip_drawClosedListOfPoints.cc. Connects the extreme points
version: 05.16.02
**************************************************************************/
void ip_drawClosedListOfPoints(int NumberOfPoints, ip_Point *pPointList, 
			       ip_Image<ip_ColorElement8u> &Image, 
			       ip_ColorElement8u color=ip_GREEN)
{
  for(int i=0; i<NumberOfPoints-1; i++)
    ip_bresenham((int)(pPointList+i)->m_rX, (int)(pPointList+i)->m_rY, 
		 (int)(pPointList+i+1)->m_rX, (int)(pPointList+i+1)->m_rY, 
		 Image, color);

  ip_bresenham((int)(pPointList+NumberOfPoints-1)->m_rX, 
	       (int)(pPointList+NumberOfPoints-1)->m_rY, 
	       (int)pPointList->m_rX, (int)pPointList->m_rY, 
	       Image, color);

}  /* ip_drawClosedListOfPoints() */




/**************************************************************************
ip_safeDraw
version: 06.06.02
**************************************************************************/
void ip_safeDraw(ip_Point *pP1, ip_Point *pP2, 
					  ip_Image<ip_ColorElement8u> *pImage, 
					  ip_ColorElement8u color=ip_GREEN)
{

  ip_Point auxPoint;
  int rows = pImage->nbLines();
  int cols = pImage->nbColumns();

  if( ip_inImage(pP1,rows,cols)){
	 if(ip_inImage(pP2,rows,cols) )
		{
		  ip_bresenham(pP1, pP2, pImage, color);
		}   
	 else 
		{   
		  ip_extremePoint(pP1, pP2, &auxPoint, rows, cols);
		  ip_bresenham(pP1, &auxPoint, pImage, color);
		}
  }
  else if(ip_inImage(pP2,rows,cols) )
	 {    
		ip_extremePoint(pP2, pP1, &auxPoint, rows, cols);
		ip_bresenham(&auxPoint, pP2, pImage, color);
	 }
  
}


/**************************************************************************
ip_drawClosedListOfPoints.cc. Overloaded function using pointer to image
version: 06.06.02
**************************************************************************/
void ip_drawClosedListOfPoints(int NumberOfPoints, ip_Point *pPointList, 
			       ip_Image<ip_ColorElement8u> *pImage, 
			       ip_ColorElement8u color=ip_GREEN)
{
  for(int i=0; i<NumberOfPoints-1; i++)
    ip_safeDraw(pPointList+i,pPointList+i+1, pImage, color);         			     
  ip_safeDraw(pPointList+NumberOfPoints-1,pPointList,pImage,color);

}  /* ip_drawClosedListOfPoints() */




/**************************************************************************
ip_drawOpenListOfPoints.cc Does not connect the extreme points
version: 05.16.02
**************************************************************************/
void ip_drawOpenListOfPoints(int NumberOfPoints, ip_Point *pPointList, 
			     ip_Image<ip_ColorElement8u>  &Image, 
			     ip_ColorElement8u color=ip_GREEN)
{
  for(int i=0; i<NumberOfPoints-1; i++)
    ip_bresenham((int)(pPointList+i)->m_rX, (int)(pPointList+i)->m_rY, 
		 (int)(pPointList+i+1)->m_rX, (int)(pPointList+i+1)->m_rY, 
		 Image, color);

} /* ip_drawOpenListOfPoints() */


/**************************************************************************
ip_drawOpenListOfPoints.cc Overloaded function using pointer to image
version: 05.16.02
**************************************************************************
void ip_drawOpenListOfPoints(int NumberOfPoints, ip_Point *pPointList, 
			     ip_Image<ip_ColorElement8u>  *pImage, 
			     ip_ColorElement8u color=ip_GREEN)
{
  for(int i=0; i<NumberOfPoints-1; i++)
    if( ip_inImage(pPointList[i], pImage->nbLines(), pImage->nbColumns()) &&
	ip_inImage(pPointList[i+1], pImage->nbLines(), pImage->nbColumns()))
      ip_bresenham((int)(pPointList+i)->m_rX, (int)(pPointList+i)->m_rY, 
		   (int)(pPointList+i+1)->m_rX, (int)(pPointList+i+1)->m_rY, 
		   pImage, color);

} ip_drawOpenListOfPoints() */



/**************************************************************************
ip_drawOpenListOfPoints.cc Overloaded function using pointer to image
version: 05.16.02
**************************************************************************/
void ip_drawOpenListOfPoints(int NumberOfPoints, ip_Point *pPointList, 
			     ip_Image<ip_ColorElement8u>  *pImage, 
			     ip_ColorElement8u color=ip_GREEN)
{
  for(int i=0; i<NumberOfPoints-1; i++)
    ip_safeDraw(pPointList+i,pPointList+i+1, pImage, color);

} /* ip_drawOpenListOfPoints() */




/**************************************************************************
ip_bresenham.cc
version: 05.16.02
**************************************************************************/
void ip_bresenham(int x1, int y1, int x2, int y2, 
		  ip_Image<ip_ColorElement8u>  &Image, 
		  ip_ColorElement8u color=ip_GREEN)
{

  int x, y, delta_x, delta_y, s1, s2, temp, change, error, i;

  // initialize variables
  x=x1;
  y=y1;
  delta_x=abs(x2-x1);
  delta_y=abs(y2-y1);
  s1=ip_sign(x2-x1);
  s2=ip_sign(y2-y1);

  // exchange delta_x y delta_y depending on the line slope 
  if(delta_y>delta_x)
  {
    temp=delta_x;
    delta_x=delta_y;
    delta_y=temp;
    change=1;
  }
  else change=0;

  // initialize the error term to compensate for a non-zero intercept 
  error=2*delta_y-delta_x;

  // main loop
  for(i=0; i<delta_x; i++)
  {	 	  
    // draw (x,y) in the desired color
    Image(y,x)=color;

    while(error>=0)
    {
      if(change==1)
	x=x+s1;
      else
	y=y+s2;
      error = error-2*delta_x;
    }
    if(change==1)
      y=y+s2;
    else
      x=x+s1;
    error = error+2*delta_y;

  } /* del for */

} /* ip_bresenham() */


/**************************************************************************
ip_bresenham.cc Overloaded function using pointer to image
version: 05.16.02
**************************************************************************/
void ip_bresenham(int x1, int y1, int x2, int y2, 
		  ip_Image<ip_ColorElement8u>  *pImage, 
		  ip_ColorElement8u color=ip_GREEN)
{

  int x, y, delta_x, delta_y, s1, s2, temp, change, error, i;

  // initialize variables
  x=x1;
  y=y1;
  delta_x=abs(x2-x1);
  delta_y=abs(y2-y1);
  s1=ip_sign(x2-x1);
  s2=ip_sign(y2-y1);

  // exchange delta_x y delta_y depending on the line slope 
  if(delta_y>delta_x)
  {
    temp=delta_x;
    delta_x=delta_y;
    delta_y=temp;
    change=1;
  }
  else change=0;

  // initialize the error term to compensate for a non-zero intercept 
  error=2*delta_y-delta_x;

  // main loop
  for(i=0; i<delta_x; i++)
  {	 	  
    // draw (x,y) in the desired color
    pImage->setVal(y, x, color);  

    while(error>=0)
    {
      if(change==1)
	x=x+s1;
      else
	y=y+s2;
      error = error-2*delta_x;
    }
    if(change==1)
      y=y+s2;
    else
      x=x+s1;
    error = error+2*delta_y;

  } /* del for */

} /* ip_bresenham() */


/**************************************************************************
ip_bresenham.cc
version: 05.22.02
**************************************************************************/
void ip_bresenham(ip_Point *pPoint1, ip_Point *pPoint2,
		  ip_Image<ip_ColorElement8u>  &Image, 
		  ip_ColorElement8u color=ip_GREEN)
{
  int x1 = (int)(pPoint1->m_rX);
  int y1 = (int)(pPoint1->m_rY);
  int x2 = (int)(pPoint2->m_rX);
  int y2 = (int)(pPoint2->m_rY);
  ip_bresenham(x1, y1, x2, y2, Image, color);
}


/**************************************************************************
ip_bresenham.cc. Overloaded function using pointer to image
version: 05.22.02
**************************************************************************/
void ip_bresenham(ip_Point *pPoint1, ip_Point *pPoint2,
		  ip_Image<ip_ColorElement8u>  *pImage, 
		  ip_ColorElement8u color=ip_GREEN)
{
  int x1 = (int)(pPoint1->m_rX);
  int y1 = (int)(pPoint1->m_rY);
  int x2 = (int)(pPoint2->m_rX);
  int y2 = (int)(pPoint2->m_rY);
  ip_bresenham(x1, y1, x2, y2, pImage, color);
}


}

