#ifndef BICV_TRANSFORMED_ELLIPSE_TEMPLATE_INC
#define BICV_TRANSFORMED_ELLIPSE_TEMPLATE_INC

#include "general.h"
#include "bf_RandomVariable.h"
#include "bf_RandomGenerator.h"
#include "bf_DistributionReal.h"

#include "bf_RVFunctions.h"

// open cv
#include "cv.h"

#include "ip_ImageFeatureFunctions.h"
#include "ip_Ellipse.h"
#include "ipcv_ContourTemplate.h"
#include "bi_ContourMeasurer.h"
#include "ip_PlanarTransform.h"



using namespace ImageProcessing;

namespace Torch {

  //-----

  /**       
      @author Jean-marc Odobez (Jean-Marc.Odobez@idiap.ch)
      @author Daniel Gatica-Perez (gatica@idiap.ch)
  */

  /*
  
    Object is an ellipse
   
     Transformation are planar transformation
  
    Transform(shape) = transform the parameters of the shape 
  
    Parameters of the configuration : 
        - should be provided inside a vector of reals.
          This vector may contain other parameters not related
           to the configuration. Nevertheless, the "useful"
           parameters (i.e. the ones that define the configuration) 
           should be placed regularly in this vector, the first one
           being at the 0 position.
        - when providing the configuration parameters (cf setTrasnformReal)
          the "step size" between two useful parameters in the input 
          vector can be set with the setStepSet() function.
        - when recovering configuration parameters (getTransformReal)
          the "step size" between two places to store two consecutive parameters
          can be specified with the setStepGet() function.
	- inside means whether 

  */

  class bicv_TransformedEllipseTemplate : virtual public bi_ContourMeasurer 
  {
    public:
    ip_Ellipse           *m_pEllipseTemplate;     //might allocate memory here
    ip_Ellipse           *m_pEllipseCurrent;      //allocates memory here
    ip_PlanarTransform   *m_pTransform;           //only the pointer, no allocate
  
    private:
    bool                  m_bTemplateAllocated;  // want memory for the template?
    int                   m_iStepGet;
    ip_Pixel              m_cCenter;             // centroid of ellipse
    bool                  m_bCurrentInside;      // is current ellipse "inside" the image?


    public:

    //-----
    bicv_TransformedEllipseTemplate();

    //-----
    bicv_TransformedEllipseTemplate(ip_PlanarTransform *pTransf,
				    bool AllocateMemTemp=false);

    //-----
    bicv_TransformedEllipseTemplate(ip_Ellipse *pEllipseTemplate,
				    ip_PlanarTransform *pTransf,	    
				    bool AllocateMemTemp=false,
				    bool with_center=false);

    //----
    virtual void  create(ip_Ellipse *pEllipseTemplate,
			 ip_PlanarTransform *pTransf,	    
			 bool AllocateMemTemp=false,
			 bool with_center=false);
       
    //----
    virtual void setStepSet(int step) { m_pTransform->m_iStepRead=step;}
 
    //----
    virtual void setStepGet(int step) { m_iStepGet=step; }
    
    //----
    // compute centroid of template ellipse
    virtual ip_Pixel computeCenter(void);

    //----
    // retrieve the center of gravity of template ellipse
    virtual ip_Pixel getCenter(void);

    //----
    // center the template ellipse around centroid
    virtual void center(void);

    //---- 
    // set the bounding box template with initialisation
    virtual void  setEllipse(ip_Ellipse *pEllipseTemplate,
			     bool with_center);

    //---- 
    virtual bool isReal() { return true;}

    //---- 
    virtual int nbParam()
    { if(m_pTransform!=NULL) return m_pTransform->m_iNbParams; 
      else return 0;
    }

    //----
    // assumes output ellipse pOut has been already allocated
    virtual void transformEllipse(ip_Ellipse *pIn, ip_Ellipse *pOut);

    //----
    // inherited from bi_ShapeTransformTemplate
    virtual void transform(void);
    
    //-----
    // inherited from bi_ShapeTransformTemplate
    virtual inline void getTransformReal(real *pTrPars)
    {m_pTransform->getParams(pTrPars,m_iStepGet);}

    //----- dgp
    // inherited from bi_ShapeTransformTemplate
    virtual inline real *getTransformReal(void)
    {return m_pTransform->getPtrToParams();}


    //-----
    // inherited from bi_ShapeTransformTemplate
    virtual inline void setTransformReal(real *pTrPars)
    {m_pTransform->setParams(pTrPars);}

    //----
    // inherited from ContourMeasurer: version for ip_Image
    virtual bool computeContourMeasurements(ip_Image <ip_ColorElement8u> *pImage,
					    ipcv_ContourTemplate *pCT);

    //----
    // inherited from ContourMeasurer: version for IplImage
    virtual bool computeContourMeasurements(IplImage **pImage,
					    ipcv_ContourTemplate *pCT);

    // ----
    virtual void  draw(ip_Image<ip_ColorElement8u> *pImage,
		       ip_ColorElement8u color,
		       ipcv_ContourTemplate *pCT,
		       bool drawCurrentContour=true,
		       bool drawLines=false,
		       bool drawFeatures=false);

    //----    
    virtual void freeMemory();

    //----
    virtual ~bicv_TransformedEllipseTemplate() 
    {freeMemory();printf("out bicv_TransformedEllipseTemplate\n");fflush(stdout);}

  };
   
}

#endif  // BICV_TRANSFORMED_ELLIPSE_TEMPLATE_INC
