#ifndef BF_RANDOM_INC
#define BF_RANDOM_INC

#include "general.h"


#define bf_RandomGenerator  bf_Random

namespace Torch {

/** BF_Random class which contains NO static random methods.
    
    These methods are based on a uniform random generator,
    named "Mersenne Twister", available at:
    http://www.math.keio.ac.jp/matumoto/emt.html.
    Copyright Makoto Matsumoto and Takuji Nishimura.
    (Have a look inside the implementation file for details).

    The random generator can be initialized with the manualSeed() method.
    Otherwise, it will be automatically initialized with a seed based
    on the current computer clock.

    @author Ronan Collobert (collober@iro.umontreal.ca)
*/
class bf_Random
{
  public:

    // Internal variables for the Mersenne Twister generator
     static const int n = 624;
     static const int m = 397;
     unsigned long *state; /* the array for the state vector  */
     int left;
     int initf;
     unsigned long *next;

    // Internal variables for the normal distribution generator
     real normal_x;
     real normal_y;
     real normal_rho;
     bool normal_is_valid;

     bf_Random();
     bf_Random(unsigned long s);

    // Internal method for the Mersenne Twister generator
     void nextState();

    /// Initializes the random number generator with the computer clock.
     void seed();

    /// Initializes the random number generator with the given long "s".
     void manualSeed(unsigned long s);

    /// Generates a uniform 32 bits integer.
     unsigned long random();

    /// Generates a uniform random number on [0,1[.
     real uniform();

    /// Returns in #indices# #n_indices# shuffled. (between 0 and #n_indices-1#).
     void getShuffledIndices(int *indices, int n_indices);

    /// Shuffles tabular, which contains #n_elems# of size #size_elem#.
     void shuffle(void *tabular, int size_elem, int n_elems);

    /// Generates a uniform random number on [a,b[ (b>a).
     real boundedUniform(real a, real b);

    /** Generates a random number from a normal distribution.
        (With mean #mean# and standard deviation #stdv >= 0#).
    */
     real normal(real mean=0, real stdv=1);

    /** Generates a random number from an exponential distribution.
        The density is $p(x) = lambda * exp(-lambda * x)$, where
        lambda is a positive number.
    */
     real exponential(real lambda);

    /** Returns a random number from a Cauchy distribution.
        The Cauchy density is $p(x) = sigma/(pi*(sigma^2 + (x-median)^2))$
    */
     real cauchy(real median=0, real sigma=1);

    /** Generates a random number from a log-normal distribution.
        (#mean > 0# is the mean of the log-normal distribution
        and #stdv# is its standard deviation).
    */
     real logNormal(real mean, real stdv);

    /** Generates a random number from a geometric distribution.
        It returns an integer #i#, where $p(i) = (1-p) * p^(i-1)$.
        p must satisfy $0 < p < 1$.
    */
     int geometric(real p);

    /// Returns true with probability $p$ and false with probability $1-p$ (p > 0).
     bool bernouilli(real p=0.5);


     inline void set(){}
     inline void release(){}
     inline real gaussian_01(){ return normal();}



     ~bf_Random();
};

}

#endif
