#ifndef BF_PARTICLEDISTRIBUTION_INC
#define BF_PARTICLEDISTRIBUTION_INC

#include "general.h"
#include "bf_RandomGenerator.h"
#include "bf_RandomVariable.h"
#include "Distribution.h"
#include "bf_Distribution.h"

namespace Torch {

//-----

/** Provides a definition of a particle set distribution
    (a set of weighted samples)

    The memory for a ParticleDistribution has to be
    defined somewhere else

    The class is not responsible for checking that all of 
    its elements (particles) are either of the same type
    or of the same dimensions.

    @author Jean-Marc Odobez (Jean-Marc.Odobez@idiap.ch)
    @author Daniel Gatica-Perez (gatica@idiap.ch)
*/

  class bf_ParticleDistribution 
  {
    public:
      
      bf_RandomVariable   **m_pSampleSet;
      real                 *m_pWeights;
      int                   m_iNumberOfSamples;

      //auxiliary variables for resampling
      real                 *m_pCumWeights;
      real                  m_dTotalCumWeight;


      // generator that should be used by the sampling function
      // should be inherited from Distribution
      bf_RandomGenerator   *m_pRng; 

      /// creates a particle distribution pointing to N samples
      bf_ParticleDistribution(int N, bf_RandomGenerator *rng);
      bf_ParticleDistribution(bf_RandomVariable **pX, int N, bf_RandomGenerator *rng);
      bf_ParticleDistribution(bf_RandomVariable *pX, int N, bf_RandomGenerator *rng); 


      // creates the memory necessary for managing for N particles
  private:
      void create(int N,bf_RandomGenerator *rng);

  public:
      //-----
  
      /** Initialize the conditional distribution
	  All objects *MUST* call the #init()#
	  function of their parents.
      */
      virtual void init(void);
  

      //-----

      // allow to reset the values of the particles
      // ( should be consistent with the content of
      //   the particles )
      virtual void reset(bf_SampleDist *sampler);
  
  
      //-----
      // return the index of the particle with highest weight
      virtual int getMode();
  
      //-----

      // draw one sample from the particle set. memory for *pX should exist somewhere
      virtual void sampleParticle(bf_RandomVariable **pX);

      // draw one sample from the particle set. memory for *pX should exist somewhere
      // deliver the corresponding index (supposes it exists)
      virtual void sampleParticleIndex(bf_RandomVariable **pX,int *index);

      // draw N samples from the particle set. memory for *pX and the samples should exist somewhere
      virtual void sampleParticleN(bf_RandomVariable **pX, int N);
  
      //resampling auxiliary functions
      virtual int SampleBinarySearch(void);

      // PENDING! evaluate: necessary for importance sampling: expensive computationally (kernel density estimation)
      virtual real evaluate(bf_RandomVariable *pX);

      // set the seed used for sampling
      virtual void setSeed(unsigned long seed);

      // set the seed used for sampling
      //virtual long getSeed(void);

      //-----
      virtual ~bf_ParticleDistribution();

    
  };



}

#endif
