#include "bf_MixedParticleDistribution.h"

namespace Torch {

  //---     Constructors 
  
  /// creates a particle distribution pointing to N samples
  bf_MixedParticleDistribution::bf_MixedParticleDistribution(int N, int NbOfRealElements,
							     int *_pSizeOfElements,
							     int _SizeOfIntElement,
							     bf_RandomGenerator *rng) :
    bf_ParticleDistribution(N,rng) {
    create(N,NbOfRealElements,_pSizeOfElements,_SizeOfIntElement);
  }
  
  /// creates a particle distribution pointing to N samples
  bf_MixedParticleDistribution::bf_MixedParticleDistribution(int N, int NbOfRealElements,
							     int _SizeOfElements,
							     int _SizeOfIntElement,
							     bf_RandomGenerator *rng) :
    bf_ParticleDistribution(N,rng) {
    int  i,*S=NULL;
    
    if(NbOfRealElements>0)
      S= (int *)malloc (NbOfRealElements*sizeof(int));
    for(i=0;i<NbOfRealElements;i++)
      S[i]=_SizeOfElements;

    //   m_iNumberOfSamples=N;
    create(N,NbOfRealElements,S,_SizeOfIntElement);

    if(NbOfRealElements>0)
      free(S);
  }


  
  // creates the memory necessary for managing  N particles
  // WARNING : IT ONLY CREATES THE ADDITIONAL MEMORY
  // (not the one necessary for the base class)

  void bf_MixedParticleDistribution::create(int N,int NbOfRealElements,
					    int *_pSizeOfElements,
					    int _SizeOfIntElement){
    int i,j,realsize;

    /*
    printf("N=%d\n",N);
    printf("Nb of real elements =%d\n",NbOfRealElements);
    printf("_pSizeOfElements[0]=%d\n",_pSizeOfElements[0]);
    printf("Nb of int elements =%d\n",_SizeOfIntElement);
    */

    m_iNbOfRealElements=NbOfRealElements;
    if(N!=m_iNumberOfSamples){
      printf("\n WARNING ERROR : the number of samples should be the same as N\n");
      fflush(stdout);
      exit(0);
    }

    
    if(m_iNbOfRealElements>0)
      m_pSizeOfRealElements=(int *) malloc(m_iNbOfRealElements * sizeof(int));
    else
      m_pSizeOfRealElements=NULL;
    
    realsize=0;
    for(i=0;i<m_iNbOfRealElements;i++){
      m_pSizeOfRealElements[i]=_pSizeOfElements[i];
      realsize+=_pSizeOfElements[i];
    }
    
    if(realsize>0 && N>0){
      m_iSizeOfStorageReal=realsize*N;
      m_pStorageReal= (real *) malloc(realsize*N*sizeof(real));
    }
    else {
      m_iSizeOfStorageReal=0;
      m_pStorageReal=NULL;
    }
      
    
    m_iSizeOfIntElement=_SizeOfIntElement;
    if(_SizeOfIntElement>0){
      m_iSizeOfStorageInt=_SizeOfIntElement*N;
      m_pStorageInt=(int *)malloc(_SizeOfIntElement*N*sizeof(int));
    }
    else {
      m_iSizeOfStorageInt=0;
      m_pStorageInt=NULL;
    }
    
    if(N>0){
      // does not work with xalloc. Why ?
      // even after performing the initialisation of the 
      // random variable, addToData produces a segmentation fault
      //m_pRVTable=(bf_RandomVariable *)xalloc(N*sizeof(bf_RandomVariable));
      m_pRVTable = new bf_RandomVariable [N];
    }
    else
      m_pRVTable=NULL;


    // Memory allocation done
    // Now we need to organize the random variables
    
    real *pr=m_pStorageReal;
    int  *pi=m_pStorageInt;
    
    // IMPORTANT : NOTE THAT m_pSampleSet HAS ALREADY BEEN ALLOCATED
    for(i=0;i<N;i++){
      m_pSampleSet[i] = &m_pRVTable[i];
      m_pWeights[i] = (real)1/N;
      m_pCumWeights[i] = (real)i/N;
      m_dTotalCumWeight = 1.0;

      for(j=0;j<m_iNbOfRealElements;j++){
	//	printf("T 4 %d %d \n",i,j); fflush(stdout);
	if(m_pSizeOfRealElements[j]>0)
	  m_pSampleSet[i]->addToData(pr);
	pr+=m_pSizeOfRealElements[j];
      }
      //     printf("T 5 %d  \n",i); fflush(stdout);
      if(m_iSizeOfIntElement>0){
	m_pSampleSet[i]->addToData((real *)pi);
	pi+=m_iSizeOfIntElement;
      }
    }

    resetZero();

  } // end of create function
  
 
  

  //-----
  // OK : lets define a resetZero function

  void bf_MixedParticleDistribution::resetZero(){
    int i;
    real *p;
    int  *pi;

    p=m_pStorageReal;
    for(i=0;i<m_iSizeOfStorageReal;i++){  *p++=0.; }
    pi=m_pStorageInt;
    for(i=0;i<m_iSizeOfStorageInt;i++){   *pi++=0; }
  }


  //-----
  // to initialize the given element [0,NbOfRealElements-1]
  void bf_MixedParticleDistribution::getSample(real *values,int particle,int element){
    ListReal           *L;
    real               *p;
    int                 i;

    if(0<=particle && particle<m_iNumberOfSamples){
      L=&m_pSampleSet[particle]->m_cData;;
      if(element>=0 && element <L->n_nodes){
	p=(real *)L->nodes[element];
	for(i=0;i<m_pSizeOfRealElements[element];i++)
	  *values++ = *p++;
      }
    }
  }

  
  //-----
  // to initialize the given element [0,NbOfRealElements-1]
  void bf_MixedParticleDistribution::setSample(real *values,int particle,int element){
    real               *p;

    if(0<=particle && particle<m_iNumberOfSamples){
      if(element>=0 && element <m_iNbOfRealElements){
	int                 i;
	p=m_pSampleSet[particle]->m_cData.nodes[element];
	for(i=0;i<m_pSizeOfRealElements[element];i++)
	  *p++=*values++;
      }
      else { // set all elements
	int i;
	element=m_iNbOfRealElements;
	while(element>0){
	  p=m_pSampleSet[particle]->m_cData.nodes[element];
	  for(i=0;i<m_pSizeOfRealElements[element];i++)
	    *p++=values[i];
	  element--;
	}
      }
    }
  }

  
  //-----
  void bf_MixedParticleDistribution::setSampleInt(int *values,int particle){
    int                *pi;

    if(0<=particle && particle<m_iNumberOfSamples){
      if(m_iSizeOfIntElement>0){
	// skip reals
	int element=m_iNbOfRealElements;
	pi=(int *)m_pSampleSet[particle]->m_cData.nodes[element];
	for(particle=0;particle<m_iSizeOfIntElement;particle++)
	  *pi++=*values++;
      }
    }
	
  }


  //-----
  void bf_MixedParticleDistribution::setAllSample(real *values,int element){
    int i;
    for(i=0;i<m_iNumberOfSamples;i++)
      setSample(values,i,element);
  }

  //-----
  void bf_MixedParticleDistribution::setAllSampleInt(int *values){
    int i;
    for(i=0;i<m_iNumberOfSamples;i++)
      setSampleInt(values,i);
  }

  //--------------------------------
  // To get some statistics on states
  void bf_MixedParticleDistribution::getMeanVariance(real *mean,real *variance,int element){
    int particle;
    int i,size_e;
    real TotalWeight,W,*p;

    if(element>=0 && element <m_iNbOfRealElements){
      // zero mean and variance
      size_e=m_pSizeOfRealElements[element];
      for(i=0;i<size_e;i++){
	mean[i]=0.;
	variance[i]=0;
      }
      TotalWeight=0.;
      
      // compute sum(value) and sum(value^2)
      for(particle=0;particle<m_iNumberOfSamples;particle++){
	W=m_pWeights[particle];
	// look for element
	p=(real *)m_pSampleSet[particle]->m_cData.nodes[element];
	// add values
	for(i=0;i<size_e;i++){
	  mean[i]+= W * *p;
	  variance[i]+=W * *p * *p; p++;
	}
	TotalWeight+=W;
      }

      // compute mean and variance
      for(i=0;i<size_e;i++){
	mean[i]/=TotalWeight;
	variance[i]=variance[i]/TotalWeight-mean[i]*mean[i];
      }
    } // if (element>=0...

  }


  //-----
  // OK : and the display function of one particle

  void bf_MixedParticleDistribution::displaySample(int i,char *com){
    int j,k;
    real               *p;
    int                *pi;

    if(i<m_iNumberOfSamples)
    {
      //printf("Sample %5d (%s) : weight = %7.5f \n",i,com,m_pWeights[i]);
      printf("Sample %5d (%s) : weight = %e \n",i,com,m_pWeights[i]);
      for(j=0;j<m_iNbOfRealElements;j++){
	printf("          ");
	p=(real *)m_pSampleSet[i]->m_cData.nodes[j];
	for(k=0;k<m_pSizeOfRealElements[j];k++)
	  printf("%8.2f",p[k]);
	printf("\n");
      }
      if(m_iSizeOfIntElement>0){
	printf("    int = ");
	pi=(int *)m_pSampleSet[i]->m_cData.nodes[m_iNbOfRealElements];
	for(k=0;k<m_iSizeOfIntElement;k++)
	  printf("%8d",pi[k]);
	printf("\n");
      }
    }
  }
  
  
  
  //-----
  // OK : and a display function

  void bf_MixedParticleDistribution::display(char *com){
    int i,j,k;
    real               *p;
    int                *pi;

    printf("Particle distribution %s  \n",com);
    for(i=0;i<m_iNumberOfSamples;i++){
      printf("Sample %5d : weight = %7.5f \n",i,m_pWeights[i]);
      for(j=0;j<m_iNbOfRealElements;j++){
	printf("          ");
	p=(real *)m_pSampleSet[i]->m_cData.nodes[j];
	for(k=0;k<m_pSizeOfRealElements[j];k++)
	  printf("%8.2f",p[k]);
	printf("\n");
      }
      if(m_iSizeOfIntElement>0){
	printf("    int = ");
	pi=(int *)m_pSampleSet[i]->m_cData.nodes[m_iNbOfRealElements];
	for(k=0;k<m_iSizeOfIntElement;k++)
	  printf("%8d",pi[k]);
	printf("\n");
      }
    }
  }   

  //-----
  bf_MixedParticleDistribution::~bf_MixedParticleDistribution(){

    if(m_pSizeOfRealElements!=NULL)
      free(m_pSizeOfRealElements);
    if(m_pStorageReal!=NULL)
      free(m_pStorageReal);
    if(m_pStorageInt!=NULL)
      free(m_pStorageInt);
    if(m_pRVTable!=NULL)
      delete [] m_pRVTable;

  }

}


